<?php
// orders/delivery/delivery.php
// فرم تحویل سفارش برای راننده (بدون لاگین) با لینک امن شبیه اپلیکیشن اسنپ

require __DIR__.'/../includes/config.php';
require __DIR__.'/../includes/db_connect.php';
require __DIR__.'/../includes/jdf.php';

// اگر به هر دلیل در config تعریف نشده بود، یک مقدار موقت (ولی بهتره حتما در config تنظیم شود)
if (!defined('DELIVERY_LINK_SECRET')) {
    define('DELIVERY_LINK_SECRET', 'CHANGE_ME_DELIVERY_SECRET_123456');
}

// ساخت توکن امن بر اساس order_uid (همخوان با لینک‌سازی در پنل سر باربر)
function make_delivery_token($order_uid) {
    return hash_hmac('sha256', $order_uid, DELIVERY_LINK_SECRET);
}

function e($s){
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}

function nf($n){
    return number_format((float)$n, 0, '.', ',');
}

function toJalaliEn($d){
    if(!$d || strlen($d) < 10) return '';
    list($y,$m,$day) = explode('-', substr($d,0,10));
    $j = gregorian_to_jalali((int)$y,(int)$m,(int)$day,'/');
    return strtr($j,[
        '۰'=>'0','۱'=>'1','۲'=>'2','۳'=>'3','۴'=>'4',
        '۵'=>'5','۶'=>'6','۷'=>'7','۸'=>'8','۹'=>'9'
    ]);
}

// مپ وضعیت‌ها برای نمایش
$status_map = [
    'new'               => '🕓 ثبت شده (در انتظار بررسی)',
    'assigned_to_buyer' => '📌 ارجاع به مسئول خرید',
    'purchased'         => '🛒 خریداری شده',
    'handover_to_head'  => '📦 تحویل به سر باربر',
    'out_for_delivery'  => '🚚 در حال ارسال',
    'delivered'         => '✅ تحویل داده شده',
    'failed'            => '❌ تحویل ناموفق',
];

$code        = trim($_GET['code']  ?? '');
$token       = trim($_GET['token'] ?? '');
$okFlag      = isset($_GET['ok']) ? 1 : 0;

$order       = null;
$order_items = [];
$proofs      = [];
$errors      = [];
$success     = '';

/* ============================
   1) اعتبارسنجی اولیه لینک
============================ */

if ($code === '' || $token === '') {
    $errors[] = 'لینک تحویل معتبر نیست (پارامترهای ناقص).';
} else {

    // واکشی سفارش بر اساس tracking_code
    $stmt = $mysqli->prepare("SELECT * FROM orders WHERE tracking_code=? LIMIT 1");
    if (!$stmt) {
        http_response_code(500);
        $errors[] = 'خطای پایگاه داده (آماده‌سازی سفارش).';
    } else {
        $stmt->bind_param('s', $code);
        $stmt->execute();
        $order = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        if (!$order) {
            $errors[] = 'سفارشی با این کد پیگیری پیدا نشد.';
        } else {
            $order_uid = $order['order_uid'] ?? '';
            if ($order_uid === '') {
                $errors[] = 'شناسه داخلی سفارش نامعتبر است.';
            } else {
                $expected = make_delivery_token($order_uid);
                if (!hash_equals($expected, $token)) {
                    $errors[] = 'لینک تحویل معتبر نیست یا منقضی شده است.';
                    $order = null;
                }
            }
        }
    }

    // اگر سفارش معتبر است، اقلام و مدارک قبلی را هم واکشی کن
    if ($order) {
        $order_id = (int)$order['id'];

        // اقلام سفارش
        $stmt = $mysqli->prepare("SELECT * FROM order_items WHERE order_id=? ORDER BY id ASC");
        if ($stmt) {
            $stmt->bind_param('i', $order_id);
            $stmt->execute();
            $resI = $stmt->get_result();
            while($row = $resI->fetch_assoc()){
                $order_items[] = $row;
            }
            $stmt->close();
        }

        // مدارک تحویل قبلی
        $chk = $mysqli->query("SHOW TABLES LIKE 'order_delivery_proofs'");
        if ($chk && $chk->num_rows > 0) {
            $stmt = $mysqli->prepare("SELECT * FROM order_delivery_proofs WHERE order_id=? ORDER BY id DESC");
            if ($stmt) {
                $stmt->bind_param('i', $order_id);
                $stmt->execute();
                $resP = $stmt->get_result();
                while($row = $resP->fetch_assoc()){
                    $proofs[] = $row;
                }
                $stmt->close();
            }
        }
    }
}

/* ============================
   2) هندل ارسال فرم راننده
============================ */

if ($order && $_SERVER['REQUEST_METHOD'] === 'POST') {

    $delivery_status = $_POST['delivery_status'] ?? '';
    $driver_name     = trim($_POST['driver_name'] ?? '');
    $serial_number   = trim($_POST['serial_number'] ?? '');
    $customer_name   = trim($_POST['customer_name'] ?? ($order['customer_name'] ?? ''));
    $customer_phone  = trim($_POST['customer_phone'] ?? ($order['mobile1'] ?? ''));
    $address_snap    = trim($_POST['address_snapshot'] ?? '');
    $note            = trim($_POST['note'] ?? '');

    if ($delivery_status !== 'delivered' && $delivery_status !== 'failed') {
        $errors[] = 'وضعیت تحویل را انتخاب کنید.';
    }

    // مسیرهای ذخیره فایل (هماهنگ با پنل سر باربر)
    $upload_base = __DIR__ . '/../uploads/delivery';
    $photo_dir   = $upload_base . '/photos';
    $media_dir   = $upload_base . '/media';

    foreach ([$upload_base, $photo_dir, $media_dir] as $dir) {
        if (!is_dir($dir)) {
            @mkdir($dir, 0775, true);
        }
    }

    $photo1 = $photo2 = $photo3 = $video1 = null;
    $hasFile = false;

    // عکس‌ها
    $photo_fields = ['photo1', 'photo2', 'photo3'];
    foreach ($photo_fields as $field) {
        if (!empty($_FILES[$field]['name'])) {
            $file = $_FILES[$field];

            if ($file['error'] === UPLOAD_ERR_OK) {
                $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
                $allowed_img = ['jpg', 'jpeg', 'png', 'webp'];
                if (!in_array($ext, $allowed_img, true)) {
                    $errors[] = "فرمت فایل {$field} مجاز نیست.";
                } else {
                    $newName = 'o' . $order['id'] . '_' . time() . '_' . $field . '.' . $ext;
                    $target  = $photo_dir . '/' . $newName;
                    if (move_uploaded_file($file['tmp_name'], $target)) {
                        $hasFile = true;
                        $relPath = 'uploads/delivery/photos/' . $newName;
                        if ($field === 'photo1') $photo1 = $relPath;
                        if ($field === 'photo2') $photo2 = $relPath;
                        if ($field === 'photo3') $photo3 = $relPath;
                    } else {
                        $errors[] = "خطا در آپلود فایل {$field}.";
                    }
                }
            } elseif ($file['error'] !== UPLOAD_ERR_NO_FILE) {
                $errors[] = "خطا در آپلود فایل {$field}.";
            }
        }
    }

    // ویدیو / صوت
    if (!empty($_FILES['media1']['name'])) {
        $file = $_FILES['media1'];
        if ($file['error'] === UPLOAD_ERR_OK) {
            $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $allowed_media = ['mp4','mov','mkv','3gp','mp3','ogg','m4a','wav'];
            if (!in_array($ext, $allowed_media, true)) {
                $errors[] = "فرمت فایل ویدیو يا صوت مجاز نیست.";
            } else {
                $newName = 'o' . $order['id'] . '_' . time() . '_media.' . $ext;
                $target  = $media_dir . '/' . $newName;
                if (move_uploaded_file($file['tmp_name'], $target)) {
                    $hasFile = true;
                    $video1  = 'uploads/delivery/media/' . $newName;
                } else {
                    $errors[] = "خطا در آپلود فایل ویدیو يا صوت.";
                }
            }
        } elseif ($file['error'] !== UPLOAD_ERR_NO_FILE) {
            $errors[] = "خطا در آپلود فایل ویدیو يا صوت.";
        }
    }

    // تحویل موفق بدون مدرک ممنوع
    if ($delivery_status === 'delivered' && !$hasFile) {
        $errors[] = 'برای تحویل موفق حداقل یک عکس یا فایل ویدیو/صوت ارسال کنید.';
    }

    if (empty($errors)) {

        $order_id = (int)$order['id'];

        // چک وجود جدول مدارک تحویل
        $has_proof_table = false;
        $chk = $mysqli->query("SHOW TABLES LIKE 'order_delivery_proofs'");
        if ($chk && $chk->num_rows > 0) {
            $has_proof_table = true;
        }

        if ($has_proof_table) {
            $sql = "
                INSERT INTO order_delivery_proofs
                (order_id, item_id, serial_number, customer_name, customer_phone, address_snapshot, note,
                 photo1, photo2, photo3, video1, created_by)
                VALUES (?,?,?,?,?,?,?,?,?,?,?,?)
            ";
            $stmt = $mysqli->prepare($sql);
            if ($stmt) {
                $item_id    = null;
                $created_by = $driver_name !== '' ? $driver_name : 'driver_link';

                $stmt->bind_param(
                    'iissssssssss',
                    $order_id,
                    $item_id,
                    $serial_number,
                    $customer_name,
                    $customer_phone,
                    $address_snap,
                    $note,
                    $photo1,
                    $photo2,
                    $photo3,
                    $video1,
                    $created_by
                );
                if (!$stmt->execute()) {
                    $errors[] = 'خطا در ثبت مدارک تحویل: '.$stmt->error;
                }
                $stmt->close();
            } else {
                $errors[] = 'خطای پایگاه داده (ثبت مدارک تحویل).';
            }
        }

        // آپدیت وضعیت سفارش
        if (empty($errors) && ($delivery_status === 'delivered' || $delivery_status === 'failed')) {

            $txt = ($delivery_status === 'delivered')
                ? "تحویل موفق راننده"
                : "تحویل ناموفق راننده";

            if ($driver_name)   $txt .= " | راننده: ".$driver_name;
            if ($serial_number) $txt .= " | سریال: ".$serial_number;
            if ($note)          $txt .= " | توضیح: ".$note;

            $sql = "
                UPDATE orders
                SET shipping_status = ?,
                    status_note     = CONCAT(COALESCE(status_note,''), ?)
                WHERE id = ? LIMIT 1
            ";
            $stmt = $mysqli->prepare($sql);
            if ($stmt) {
                $stmt->bind_param('ssi', $delivery_status, $txt, $order_id);
                if (!$stmt->execute()) {
                    $errors[] = 'خطا در بروز رسانی وضعیت سفارش: '.$stmt->error;
                }
                $stmt->close();

                // به‌روزرسانی لیبل وضعیت برای نمایش
                $status_raw   = $delivery_status;
                $status_label = $status_map[$status_raw] ?? $status_raw;
                $order['shipping_status'] = $delivery_status;
            } else {
                $errors[] = 'خطای پایگاه داده (آماده‌سازی آپدیت سفارش).';
            }
        }

        // ریدایرکت فقط اگر خطا نداشتیم
        if (empty($errors)) {
            $self = strtok($_SERVER['REQUEST_URI'], '?');
            $qs = http_build_query([
                'code'  => $code,
                'token' => $token,
                'ok'    => 1
            ]);
            header("Location: ".$self.'?'.$qs);
            exit;
        }
    }
}

// برای نمایش وضعیت فعلی
$status_raw   = $order['shipping_status'] ?? 'new';
$status_label = $status_map[$status_raw] ?? $status_raw;

?><!doctype html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="utf-8">
    <title>تحویل سفارش — سلام بابا</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">

    <!-- فونت ایران یکان (مسیر را متناسب با سرور خودت تنظیم کن) -->
    <style>
    @font-face {
        font-family: 'IRANYekan';
        src: url('/orders/assets/fonts/IRANYekanRegular.woff2') format('woff2'),
             url('/orders/assets/fonts/IRANYekanRegular.woff') format('woff');
        font-weight: 400;
        font-style: normal;
        font-display: swap;
    }
    @font-face {
        font-family: 'IRANYekan';
        src: url('/orders/assets/fonts/IRANYekanMedium.woff2') format('woff2'),
             url('/orders/assets/fonts/IRANYekanMedium.woff') format('woff');
        font-weight: 500;
        font-style: normal;
        font-display: swap;
    }
    @font-face {
        font-family: 'IRANYekan';
        src: url('/orders/assets/fonts/IRANYekanBold.woff2') format('woff2'),
             url('/orders/assets/fonts/IRANYekanBold.woff') format('woff');
        font-weight: 700;
        font-style: normal;
        font-display: swap;
    }
    </style>

    <!-- استایل اپلیکیشنی شبیه اسنپ -->
    <link rel="stylesheet" href="/orders/delivery/delivery.css">
</head>
<body>

<div class="app">

    <!-- نوار بالای اپ + لوگوی چهار رنگ سلام بابا -->
    <header class="app-header">
        <div class="app-header-gradient"></div>
        <div class="app-header-content">
            <div class="brand-stripes">
                <span class="stripe b"></span>
                <span class="stripe g"></span>
                <span class="stripe y"></span>
                <span class="stripe r"></span>
            </div>
            <div class="app-title-block">
                <div class="app-title">تحویل سفارش</div>
                <?php if($order): ?>
                    <div class="app-subtitle">
                        کد پیگیری: <span class="pill pill-light"><?=e($order['tracking_code'])?></span>
                    </div>
                <?php else: ?>
                    <div class="app-subtitle">لینک تحویل سفارش</div>
                <?php endif; ?>
            </div>
        </div>
    </header>

    <main class="app-body">

        <?php if($okFlag && empty($errors)): ?>
            <div class="toast toast-success">
                <div class="toast-icon">✔</div>
                <div class="toast-text">اطلاعات تحویل با موفقیت ثبت شد.</div>
            </div>
        <?php endif; ?>

        <?php if(!empty($errors)): ?>
            <div class="toast toast-error">
                <div class="toast-icon">!</div>
                <div class="toast-text">
                    <?php foreach($errors as $er): ?>
                        <div><?=e($er)?></div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>

        <?php if(!$order): ?>
            <section class="card card-empty">
                <h2>سفارش یافت نشد</h2>
                <p>در صورت بروز مشکل با پشتیبانی سلام بابا تماس بگیرید.</p>
            </section>
        <?php else: ?>

            <!-- کارت اطلاعات سفارش (شبیه کارت اسنپ) -->
            <section class="card card-order">
                <div class="card-order-header">
                    <div class="order-id">
                        سفارش
                        <span class="pill pill-strong"><?=e($order['order_uid'])?></span>
                    </div>
                    <div class="order-status-pill"><?=$status_label?></div>
                </div>

                <div class="order-info-grid">
                    <div class="info-row">
                        <span class="info-label">مشتری</span>
                        <span class="info-value"><?=e($order['customer_name'])?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">موبایل</span>
                        <a href="tel:<?=e($order['mobile1'])?>" class="info-value info-link">
                            <?=e($order['mobile1'])?>
                        </a>
                    </div>
                    <div class="info-row">
                        <span class="info-label">تاریخ ثبت</span>
                        <span class="info-value">
                            <?=toJalaliEn($order['created_at'] ?? '')?>
                        </span>
                    </div>
                    <?php if(!empty($order['province'])): ?>
                    <div class="info-row">
                        <span class="info-label">استان</span>
                        <span class="info-value"><?=e($order['province'])?></span>
                    </div>
                    <?php endif; ?>
                    <div class="info-row info-row-address">
                        <span class="info-label">آدرس</span>
                        <span class="info-value"><?=e($order['address'])?></span>
                    </div>
                </div>

                <?php if($order_items): ?>
                    <div class="order-items">
                        <div class="items-title">اقلام سفارش</div>
                        <div class="items-list">
                            <?php $i=1; foreach($order_items as $it): ?>
                                <div class="item-row">
                                    <div class="item-main">
                                        <span class="item-index"><?= $i++ ?></span>
                                        <span class="item-name"><?=e($it['model'])?></span>
                                    </div>
                                    <div class="item-meta">
                                        <?php if(isset($it['qty'])): ?>
                                            <span class="chip chip-soft">× <?= (int)$it['qty'] ?></span>
                                        <?php endif; ?>
                                        <?php if(isset($it['cod'])): ?>
                                            <span class="chip chip-money">
                                                COD: <?=nf($it['cod'])?>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>
            </section>

            <!-- فرم ثبت تحویل (UI شبیه اسنپ/اسنپ‌فود) -->
            <section class="card card-form">
                <h2 class="section-title">ثبت وضعیت تحویل</h2>
                <p class="section-description">
                    بعد از تحویل یا عدم تحویل، وضعیت را انتخاب کنید و در صورت امکان سریال، عکس و توضیحات را ثبت نمایید.
                </p>

                <form method="post" enctype="multipart/form-data" class="delivery-form">

                    <div class="field">
                        <div class="field-label">وضعیت تحویل</div>
                        <div class="status-options">
                            <label class="status-chip">
                                <input type="radio" name="delivery_status" value="delivered"
                                    <?= (($_POST['delivery_status'] ?? '') === 'delivered') ? 'checked' : '' ?>>
                                <span class="status-chip-ui">
                                    <span class="icon">✅</span>
                                    <span>تحویل موفق</span>
                                </span>
                            </label>
                            <label class="status-chip">
                                <input type="radio" name="delivery_status" value="failed"
                                    <?= (($_POST['delivery_status'] ?? '') === 'failed') ? 'checked' : '' ?>>
                                <span class="status-chip-ui">
                                    <span class="icon">❌</span>
                                    <span>تحویل ناموفق</span>
                                </span>
                            </label>
                        </div>
                    </div>

                    <div class="field field-two-cols">
                        <div class="field">
                            <label class="field-label">نام راننده</label>
                            <input
                                class="input"
                                type="text"
                                name="driver_name"
                                value="<?=e($_POST['driver_name'] ?? '')?>"
                                placeholder="مثال: احمد - وانت نیسان">
                        </div>
                        <div class="field">
                            <label class="field-label">سریال دستگاه</label>
                            <input
                                class="input"
                                type="text"
                                name="serial_number"
                                value="<?=e($_POST['serial_number'] ?? '')?>"
                                placeholder="مثال: 1234ABC567"
                                style="direction:ltr;text-align:center;">
                        </div>
                    </div>

                    <div class="field field-two-cols">
                        <div class="field">
                            <label class="field-label">نام مشتری در محل</label>
                            <input
                                class="input"
                                type="text"
                                name="customer_name"
                                value="<?=e($_POST['customer_name'] ?? ($order['customer_name'] ?? ''))?>"
                                placeholder="نام شخص تحویل‌گیرنده">
                        </div>
                        <div class="field">
                            <label class="field-label">شماره تماس مشتری</label>
                            <input
                                class="input"
                                type="text"
                                name="customer_phone"
                                value="<?=e($_POST['customer_phone'] ?? ($order['mobile1'] ?? ''))?>"
                                placeholder="شماره موبایل"
                                style="direction:ltr;text-align:center;">
                        </div>
                    </div>

                    <div class="field">
                        <label class="field-label">توضیح کوتاه مکان / آدرس</label>
                        <textarea
                            class="textarea"
                            name="address_snapshot"
                            placeholder="مثلاً: درب غربی مجتمع، طبقه ۳، واحد ۱۲">
<?=e($_POST['address_snapshot'] ?? '')?></textarea>
                    </div>

                    <div class="field">
                        <label class="field-label">توضیحات راننده</label>
                        <textarea
                            class="textarea"
                            name="note"
                            placeholder="مثلاً: مشتری حضور داشت و دستگاه تست شد / مشتری در محل نبود / شماره پاسخ نداد و ...">
<?=e($_POST['note'] ?? '')?></textarea>
                    </div>

                    <div class="field">
                        <label class="field-label">عکس‌های تحویل (حداکثر ۳ تصویر)</label>
                        <div class="file-hints">فرمت‌های مجاز: JPG, JPEG, PNG, WEBP</div>
                        <div class="file-row">
                            <input class="file-input" type="file" name="photo1" accept="image/*">
                            <input class="file-input" type="file" name="photo2" accept="image/*">
                            <input class="file-input" type="file" name="photo3" accept="image/*">
                        </div>
                    </div>

                    <div class="field">
                        <label class="field-label">ویدیو / صوت از مشتری (اختیاری)</label>
                        <div class="file-hints">فرمت‌های مجاز: MP4, MKV, MOV, 3GP, MP3, OGG, M4A, WAV</div>
                        <input class="file-input" type="file" name="media1" accept="video/*,audio/*">
                    </div>

                    <button type="submit" class="btn-primary btn-full">
                        ✔ ثبت اطلاعات تحویل
                    </button>
                </form>
            </section>

            <!-- مدارک قبلی -->
            <?php if($proofs): ?>
                <section class="card card-proofs">
                    <h2 class="section-title">مدارک ثبت‌شده قبلی</h2>
                    <div class="proofs-grid">
                        <?php foreach($proofs as $p): ?>
                            <div class="proof-card">
                                <?php if(!empty($p['photo1'])): ?>
                                    <div class="proof-image-wrap">
                                        <img src="/orders/<?=e($p['photo1'])?>" alt="">
                                    </div>
                                <?php endif; ?>
                                <div class="proof-meta">
                                    <?php if(!empty($p['serial_number'])): ?>
                                        <div class="proof-row">
                                            <span class="tag">سریال:</span>
                                            <span><?=e($p['serial_number'])?></span>
                                        </div>
                                    <?php endif; ?>
                                    <?php if(!empty($p['note'])): ?>
                                        <div class="proof-note"><?=nl2br(e($p['note']))?></div>
                                    <?php endif; ?>
                                    <?php if(!empty($p['video1'])): ?>
                                        <div class="proof-row small">
                                            فایل ویدیو/صوت ثبت شده.
                                        </div>
                                    <?php endif; ?>
                                    <div class="proof-row small muted">
                                        <?=e($p['created_at'])?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </section>
            <?php endif; ?>

        <?php endif; ?>

    </main>

    <footer class="app-footer">
        <div class="footer-text">
            این لینک مخصوص راننده است و نیازی به ورود به سیستم ندارد.
        </div>
    </footer>

</div>

</body>
</html>
