<?php

require __DIR__.'/includes/auth.php';
require __DIR__.'/includes/config.php';
require __DIR__.'/includes/db_connect.php';
require_once __DIR__.'/includes/jdf.php';

$user     = $_SESSION['username'] ?? '';
$role     = $_SESSION['role'] ?? '';
$is_admin = ($role === 'admin');

// =========================
// Helper functions
// =========================
function nf($n){
    return number_format((int)($n ?? 0), 0, '.', ',');
}
// مسیر اصلاح شده برای نقش‌ها
if ($_SESSION['role'] === 'buyer') {
    header("Location: /orders/buyer/dashboard.php");
    exit;
}

if ($_SESSION['role'] === 'head_carrier') {
    header("Location: /orders/head/dashboard.php");
    exit;
}
// تبدیل ارقام فارسی به انگلیسی
if (!function_exists('jnum_en')) {
    function jnum_en($s){
        return strtr($s, [
            '۰'=>'0','۱'=>'1','۲'=>'2','۳'=>'3','۴'=>'4',
            '۵'=>'5','۶'=>'6','۷'=>'7','۸'=>'8','۹'=>'9'
        ]);
    }
}

// تاریخ شمسی کوتاه با ارقام انگلیسی
if (!function_exists('toJalaliEn')) {
    function toJalaliEn($date){
        if(!$date || !preg_match('/^\d{4}-\d{2}-\d{2}/', $date)) return '-';
        $date = substr($date, 0, 10);
        list($gy,$gm,$gd) = explode('-', $date);
        if(!function_exists('gregorian_to_jalali')) return $date;
        $j = gregorian_to_jalali((int)$gy,(int)$gm,(int)$gd,'/'); // مثلا ۱۴۰۳/۱۰/۲۴
        $j = jnum_en($j);
        return $j; // 1403/10/24
    }
}

// نام ماه شمسی
function jmonth_name($jm){
    $names = [
        1 => 'فروردین',
        2 => 'اردیبهشت',
        3 => 'خرداد',
        4 => 'تیر',
        5 => 'مرداد',
        6 => 'شهریور',
        7 => 'مهر',
        8 => 'آبان',
        9 => 'آذر',
        10 => 'دی',
        11 => 'بهمن',
        12 => 'اسفند',
    ];
    return $names[(int)$jm] ?? '';
}

// =========================
// تعیین ماه شمسی فعلی و لیست ۱۲ ماه اخیر
// =========================
$jyNow = (int) jnum_en(jdate('Y')); // سال شمسی فعلی
$jmNow = (int) jnum_en(jdate('m')); // ماه شمسی فعلی

// ماه انتخاب‌شده در فیلتر فروشنده
$jmonthParam = $_GET['jmonth'] ?? sprintf('%04d-%02d', $jyNow, $jmNow);
if (!preg_match('/^\d{4}-\d{2}$/', $jmonthParam)) {
    $jmonthParam = sprintf('%04d-%02d', $jyNow, $jmNow);
}
list($jySel, $jmSel) = explode('-', $jmonthParam);
$jySel = (int)$jySel;
$jmSel = (int)$jmSel;

// محاسبه بازه‌ی میلادی معادل این ماه شمسی
// شروع ماه شمسی
list($gsy, $gsm, $gsd) = jalali_to_gregorian($jySel, $jmSel, 1);
$monthStart = sprintf('%04d-%02d-%02d 00:00:00', $gsy, $gsm, $gsd);

// شروع ماه بعد شمسی
$nextJy = $jmSel === 12 ? $jySel + 1 : $jySel;
$nextJm = $jmSel === 12 ? 1         : $jmSel + 1;
list($gey, $gem, $ged) = jalali_to_gregorian($nextJy, $nextJm, 1);
$nextMonthStart = sprintf('%04d-%02d-%02d 00:00:00', $gey, $gem, $ged);

// آخرین لحظه ماه انتخاب‌شده (یک ثانیه قبل از شروع ماه بعد)
$monthEnd = date('Y-m-d 23:59:59', strtotime($nextMonthStart . ' -1 day'));

// ساخت لیست ۱۲ ماه شمسی اخیر برای سلکت
$monthOptions = [];
$jy = $jyNow;
$jm = $jmNow;
for($i=0; $i<12; $i++){
    $val = sprintf('%04d-%02d', $jy, $jm); // 1403-10
    $label = jmonth_name($jm) . ' ' . $jy; // دی 1403
    $monthOptions[] = [
        'value' => $val,
        'label' => $label
    ];
    $jm--;
    if ($jm === 0){
        $jm = 12;
        $jy--;
    }
}

// =========================
// داده‌های نمودار
// =========================
$chartLabels   = [];
$chartDeposit  = [];
$chartWithdraw = [];
$chartNet      = [];

// =========================
// حالت مدیر
// =========================
if ($is_admin) {

    // --------- آمار کلی مدیر ---------
    $totalRes = $mysqli->query("
        SELECT
          COALESCE(SUM(o.total_deposit),0) AS total_deposit,
          COUNT(DISTINCT o.id)            AS orders_count,
          COUNT(DISTINCT CASE WHEN o.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN o.id END) AS orders_30d
        FROM orders o
    ");
    $totals = $totalRes ? $totalRes->fetch_assoc() : [
        'total_deposit'  => 0,
        'orders_count'   => 0,
        'orders_30d'     => 0,
    ];

    // جمع برداشت‌ها
    $wRes = $mysqli->query("SELECT COALESCE(SUM(amount),0) AS total_withdraw FROM withdrawals");
    $wRow = $wRes ? $wRes->fetch_assoc() : ['total_withdraw' => 0];
    $totals['total_withdraw'] = (int)$wRow['total_withdraw'];

    // مانده تجمیعی
    $totals['net_balance'] = (int)$totals['total_deposit'] - (int)$totals['total_withdraw'];

    // --------- نمودار مدیر: ۱۲ ماه اخیر (گروه‌بندی میلادی، نمایش شمسی) ---------
    $chartQ = $mysqli->query("
        SELECT
          DATE_FORMAT(o.created_at, '%Y-%m-01') AS gmonth,
          COALESCE(SUM(o.total_deposit),0)      AS total_deposit,
          COALESCE(SUM(wd.total_withdraw),0)    AS total_withdraw
        FROM orders o
        LEFT JOIN (
          SELECT 
            username,
            DATE_FORMAT(created_at,'%Y-%m-01') AS gmonth,
            SUM(amount) AS total_withdraw
          FROM withdrawals
          GROUP BY username, DATE_FORMAT(created_at,'%Y-%m-01')
        ) wd ON wd.username = o.username AND wd.gmonth = DATE_FORMAT(o.created_at,'%Y-%m-01')
        GROUP BY DATE_FORMAT(o.created_at, '%Y-%m-01')
        ORDER BY gmonth ASC
    ");
    $tempRows = [];
    if ($chartQ) {
        while($row = $chartQ->fetch_assoc()){
            $tempRows[] = $row;
        }
    }
    // فقط ۱۲ ماه آخر
    $tempRows = array_slice($tempRows, -12);
    foreach($tempRows as $row){
        $gmonth = $row['gmonth']; // مثلا 2025-01-01
        list($gy,$gm,$gd) = explode('-', substr($gmonth,0,10));
        $j = gregorian_to_jalali((int)$gy,(int)$gm,(int)$gd,'/'); // ۱۴۰۳/۱۰/۱۱
        $j = jnum_en($j);
        list($jY,$jM) = explode('/', $j);
        $label = sprintf('%04d/%02d', (int)$jY, (int)$jM); // 1403/10

        $chartLabels[]   = $label;
        $chartDeposit[]  = (int)$row['total_deposit'];
        $chartWithdraw[] = (int)$row['total_withdraw'];
        $chartNet[]      = (int)$row['total_deposit'] - (int)$row['total_withdraw'];
    }

    // --------- جدول فروشنده‌ها برای مدیر ---------
    $sellers = [];
    $res = $mysqli->query("
        SELECT 
          u.username,
          u.full_name,
          COALESCE(o.total_deposit,0)  AS total_deposit,
          COALESCE(w.total_withdraw,0) AS total_withdraw,
          COALESCE(o.orders_count,0)   AS orders_count
        FROM users u
        LEFT JOIN (
          SELECT username,
                 SUM(total_deposit) AS total_deposit,
                 COUNT(*)           AS orders_count
          FROM orders
          GROUP BY username
        ) o ON o.username = u.username
        LEFT JOIN (
          SELECT username,
                 SUM(amount) AS total_withdraw
          FROM withdrawals
          GROUP BY username
        ) w ON w.username = u.username
        WHERE u.role = 'seller'
        ORDER BY u.username ASC
    ");
    if ($res) {
        while($r = $res->fetch_assoc()){
            $r['balance'] = (int)$r['total_deposit'] - (int)$r['total_withdraw'];
            $sellers[] = $r;
        }
    }

// =========================
// حالت فروشنده (غیر مدیر)
// =========================
} else {

    // --------- آمار کلی فروشنده ---------
    // مجموع بیعانه‌ها
    $stmt = $mysqli->prepare("
        SELECT COALESCE(SUM(total_deposit),0) AS total_deposit
        FROM orders
        WHERE username=?
    ");
    $stmt->bind_param('s', $user);
    $stmt->execute();
    $data = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    $seller_total_deposit = (int)($data['total_deposit'] ?? 0);

    // مجموع برداشت‌ها
    $stmt = $mysqli->prepare("
        SELECT COALESCE(SUM(amount),0) AS total_withdraw
        FROM withdrawals
        WHERE username=?
    ");
    $stmt->bind_param('s', $user);
    $stmt->execute();
    $data2 = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    $seller_total_withdraw = (int)($data2['total_withdraw'] ?? 0);

    $seller_balance = $seller_total_deposit - $seller_total_withdraw;

    // --------- نمودار فروشنده: ۱۲ ماه اخیر (گروه‌بندی میلادی، نمایش شمسی) ---------
    $chartStmt = $mysqli->prepare("
        SELECT
          DATE_FORMAT(o.created_at, '%Y-%m-01') AS gmonth,
          COALESCE(SUM(o.total_deposit),0)      AS total_deposit,
          COALESCE(SUM(wd.total_withdraw),0)    AS total_withdraw
        FROM orders o
        LEFT JOIN (
          SELECT 
            username,
            DATE_FORMAT(created_at,'%Y-%m-01') AS gmonth,
            SUM(amount) AS total_withdraw
          FROM withdrawals
          GROUP BY username, DATE_FORMAT(created_at,'%Y-%m-01')
        ) wd ON wd.username = o.username AND wd.gmonth = DATE_FORMAT(o.created_at,'%Y-%m-01')
        WHERE o.username = ?
        GROUP BY DATE_FORMAT(o.created_at, '%Y-%m-01')
        ORDER BY gmonth ASC
    ");
    $chartStmt->bind_param('s', $user);
    $chartStmt->execute();
    $chartRes = $chartStmt->get_result();
    $tempRows = [];
    while($row = $chartRes->fetch_assoc()){
        $tempRows[] = $row;
    }
    $chartStmt->close();
    $tempRows = array_slice($tempRows, -12);
    foreach($tempRows as $row){
        $gmonth = $row['gmonth'];
        list($gy,$gm,$gd) = explode('-', substr($gmonth,0,10));
        $j = gregorian_to_jalali((int)$gy,(int)$gm,(int)$gd,'/');
        $j = jnum_en($j);
        list($jY,$jM) = explode('/', $j);
        $label = sprintf('%04d/%02d', (int)$jY, (int)$jM);

        $chartLabels[]   = $label;
        $chartDeposit[]  = (int)$row['total_deposit'];
        $chartWithdraw[] = (int)$row['total_withdraw'];
        $chartNet[]      = (int)$row['total_deposit'] - (int)$row['total_withdraw'];
    }

    // --------- جدول فاکتورهای این ماه شمسی برای فروشنده ---------
    $txStmt = $mysqli->prepare("
      SELECT 
        o.id AS order_id,
        o.order_uid,
        o.invoice_no,
        o.customer_name,
        o.created_at,
        o.total_deposit,
        o.total_cod,
        o.total_amount,
        o.total_remaining,
        COALESCE(o.notes,'') AS notes
      FROM orders o
      WHERE o.username=? AND o.created_at BETWEEN ? AND ?
      ORDER BY o.created_at DESC
    ");
    $txStmt->bind_param('sss', $user, $monthStart, $monthEnd);
    $txStmt->execute();
    $txRes = $txStmt->get_result();
    $transactions = [];
    while($r = $txRes->fetch_assoc()){
        $transactions[] = $r;
    }
    $txStmt->close();
}
?>
<!doctype html>
<html lang="fa" dir="rtl">
<head>
  <meta charset="utf-8">
  <title>📊 داشبورد مالی | سلام‌بابا</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link rel="stylesheet" href="../assets/style.css">
  <style>
    body{
      font-family:'IRANSansWeb','IRANYekan',sans-serif;
      background:#f5f7fa;
      color:#0f172a;
      margin:0;
    }
    .container{
      max-width:1200px;
      margin:0 auto;
      padding:16px;
    }
    .page-header{
      display:flex;
      flex-wrap:wrap;
      align-items:center;
      justify-content:space-between;
      gap:10px;
      margin-bottom:16px;
    }
    .page-title{
      font-size:22px;
      font-weight:700;
      display:flex;
      align-items:center;
      gap:8px;
    }
    .page-title span.role-badge{
      font-size:11px;
      background:#e5f3ff;
      color:#0369a1;
      padding:3px 8px;
      border-radius:999px;
      border:1px solid #bae6fd;
    }
    .page-sub{
      font-size:13px;
      color:#6b7280;
    }

    .card{
      background:#fff;
      border-radius:16px;
      box-shadow:0 2px 8px rgba(15,23,42,0.06);
      padding:18px 20px;
      margin-bottom:18px;
      border:1px solid #e5e7eb;
    }
    .card-header{
      display:flex;
      justify-content:space-between;
      align-items:center;
      margin-bottom:10px;
    }
    .card-header h3{
      margin:0;
      font-size:16px;
      font-weight:600;
      display:flex;
      align-items:center;
      gap:6px;
    }
    .card-header small{
      font-size:12px;
      color:#6b7280;
    }

    .stats-grid{
      display:grid;
      grid-template-columns:repeat(4,minmax(0,1fr));
      gap:12px;
      margin-top:10px;
    }
    .stat-card{
      background:linear-gradient(135deg,#ffffff,#f9fafb);
      border-radius:14px;
      padding:14px 12px;
      box-shadow:0 1px 4px rgba(15,23,42,0.08);
      border:1px solid #e5e7eb;
      display:flex;
      flex-direction:column;
      gap:4px;
    }
    .stat-label{
      font-size:12px;
      color:#6b7280;
    }
    .stat-value{
      font-size:18px;
      font-weight:700;
      color:#0f172a;
    }
    .stat-chip{
      font-size:11px;
      align-self:flex-start;
      padding:2px 8px;
      border-radius:999px;
      border:1px solid #e5e7eb;
      color:#6b7280;
      background:#f9fafb;
    }
    .stat-value.positive{color:#16a34a;}
    .stat-value.negative{color:#dc2626;}

    .chart-wrapper{
      position:relative;
      min-height:260px;
    }
    #financeChart{
      width:100% !important;
      height:260px !important;
    }
    #chartEmptyMsg{
      position:absolute;
      inset:0;
      display:flex;
      align-items:center;
      justify-content:center;
      font-size:13px;
      color:#6b7280;
    }

    .table-toolbar{
      display:flex;
      flex-wrap:wrap;
      align-items:center;
      justify-content:space-between;
      gap:8px;
      margin-bottom:10px;
    }
    .toolbar-group{
      display:flex;
      flex-wrap:wrap;
      align-items:center;
      gap:6px;
    }
    .input-sm, .select-sm{
      padding:6px 8px;
      border-radius:8px;
      border:1px solid #d1d5db;
      font-size:12px;
      font-family:inherit;
      background:#fff;
      min-width:120px;
    }
    .input-sm:focus, .select-sm:focus{
      outline:none;
      border-color:#2563eb;
      box-shadow:0 0 0 2px rgba(37,99,235,0.1);
    }
    .btn-sm{
      padding:6px 10px;
      border-radius:8px;
      border:none;
      font-size:12px;
      cursor:pointer;
      font-family:inherit;
      display:inline-flex;
      align-items:center;
      gap:4px;
    }
    .btn-primary{
      background:#2563eb;
      color:#fff;
    }
    .btn-primary:hover{background:#1d4ed8;}

    .table-wrapper{
      overflow-x:auto;
      border-radius:12px;
      border:1px solid #e5e7eb;
    }
    table{
      width:100%;
      border-collapse:collapse;
      min-width:800px;
      font-size:13px;
    }
    th,td{
      padding:8px 10px;
      text-align:center;
      border-bottom:1px solid #f1f5f9;
      white-space:nowrap;
    }
    thead th{
      background:#f8fafc;
      font-weight:600;
      color:#111827;
      cursor:pointer;
      user-select:none;
      position:relative;
    }
    tbody tr:nth-child(even){
      background:#fafafa;
    }
    tbody tr:hover{
      background:#eef2ff;
    }

    .tag-badge{
      padding:2px 6px;
      border-radius:999px;
      font-size:11px;
      border:1px solid #e5e7eb;
      background:#f9fafb;
      color:#6b7280;
    }

    .btn-table{
      font-size:12px;
      padding:3px 8px;
      border-radius:6px;
      text-decoration:none;
      display:inline-flex;
      align-items:center;
      gap:3px;
    }
    .btn-table.primary{
      background:#2563eb;
      color:#fff;
    }
    .btn-table.ghost{
      background:#f3f4f6;
      color:#111827;
    }
    .badge-balance-pos{
      color:#16a34a;
      font-weight:700;
    }
    .badge-balance-neg{
      color:#dc2626;
      font-weight:700;
    }

    .pagination{
      display:flex;
      justify-content:center;
      gap:4px;
      margin-top:8px;
      flex-wrap:wrap;
    }
    .pagination button{
      border:none;
      padding:4px 8px;
      border-radius:6px;
      background:#e5e7eb;
      color:#374151;
      font-size:12px;
      cursor:pointer;
    }
    .pagination button.active{
      background:#2563eb;
      color:#fff;
      font-weight:600;
    }
    .pagination button:hover{
      background:#dbeafe;
    }

    @media(max-width:992px){
      .stats-grid{
        grid-template-columns:repeat(2,minmax(0,1fr));
      }
      table{font-size:12px;min-width:700px;}
    }
    @media(max-width:640px){
      .stats-grid{
        grid-template-columns:1fr 1fr;
      }
      .page-header{
        align-items:flex-start;
      }
      table{min-width:650px;}
    }
  </style>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
<div class="container">
  <?php require __DIR__.'/includes/header.php'; ?>

  <div class="page-header">
    <div>
      <div class="page-title">
        📊 داشبورد مالی
        <span class="role-badge">
          <?= $is_admin ? 'مدیر سیستم' : 'فروشنده' ?> — <?= htmlspecialchars($user) ?>
        </span>
      </div>
      <div class="page-sub">
        ورود و خروج پول بر اساس تاریخ شمسی، آماده توسعه برای تحلیل سود و هزینه.
      </div>
    </div>
  </div>

  <?php if ($is_admin): ?>

    <!-- ================= آمار بالا برای مدیر ================= -->
    <section class="card">
      <div class="card-header">
        <h3>📈 آمار کلی سیستم</h3>
      </div>
      <div class="stats-grid">
        <div class="stat-card">
          <div class="stat-label">جمع کل بیعانه‌های ثبت‌شده</div>
          <div class="stat-value"><?= nf($totals['total_deposit'] ?? 0) ?> <span style="font-size:11px;">تومان</span></div>
          <div class="stat-chip">همه فروشنده‌ها</div>
        </div>
        <div class="stat-card">
          <div class="stat-label">جمع کل برداشت‌ها</div>
          <div class="stat-value"><?= nf($totals['total_withdraw'] ?? 0) ?> <span style="font-size:11px;">تومان</span></div>
          <div class="stat-chip">پرداخت‌شده به فروشنده‌ها</div>
        </div>
        <div class="stat-card">
          <div class="stat-label">مانده تجمیعی فروشنده‌ها</div>
          <div class="stat-value <?= ($totals['net_balance'] ?? 0) >= 0 ? 'positive' : 'negative' ?>">
            <?= nf($totals['net_balance'] ?? 0) ?> <span style="font-size:11px;">تومان</span>
          </div>
          <div class="stat-chip">بیعانه‌ها − برداشت‌ها</div>
        </div>
        <div class="stat-card">
          <div class="stat-label">تعداد فاکتورهای ثبت‌شده</div>
          <div class="stat-value"><?= nf($totals['orders_count'] ?? 0) ?></div>
          <div class="stat-chip">۳۰ روز اخیر: <?= nf($totals['orders_30d'] ?? 0) ?></div>
        </div>
      </div>
    </section>

    <!-- ================= نمودار برای مدیر ================= -->
    <section class="card">
      <div class="card-header">
        <h3>📊 روند ماهانه ورود و خروج پول (شمسی)</h3>
        <small>لیبل‌ها به‌صورت 1403/09, 1403/10, 1403/11</small>
      </div>
      <div class="chart-wrapper">
        <canvas id="financeChart"></canvas>
        <div id="chartEmptyMsg" style="display:none;">داده‌ای برای نمودار یافت نشد.</div>
      </div>
    </section>

    <!-- ================= جدول فروشنده‌ها ================= -->
    <section class="card">
      <div class="card-header">
        <h3>👥 وضعیت مالی فروشنده‌ها</h3>
        <small>کلیک روی نام کاربری → صفحه‌ی مالی همان فروشنده</small>
      </div>

      <div class="table-toolbar">
        <div class="toolbar-group">
          <input type="text" id="sellerSearch" class="input-sm" placeholder="🔍 جستجو در نام کاربری / نام کامل">
          <select id="balanceFilter" class="select-sm">
            <option value="">همه مانده‌ها</option>
            <option value="pos">فقط مانده مثبت</option>
            <option value="neg">فقط مانده منفی</option>
            <option value="zero">فقط مانده صفر</option>
          </select>
        </div>
        <div class="toolbar-group">
          <span class="page-sub">منبع اصلی برای تحلیل سود و هزینه در نسخه‌های بعدی.</span>
        </div>
      </div>

      <div class="table-wrapper">
        <table id="sellerTable">
          <thead>
            <tr>
              <th data-type="text">👤 فروشنده</th>
              <th data-type="text">نام کامل</th>
              <th data-type="num">تعداد فاکتورها</th>
              <th data-type="num">جمع بیعانه‌ها</th>
              <th data-type="num">جمع برداشت‌ها</th>
              <th data-type="num">مانده حساب</th>
            </tr>
          </thead>
          <tbody>
          <?php foreach($sellers as $r): 
            $remain = (int)$r['balance'];
            $cls = $remain >= 0 ? 'badge-balance-pos' : 'badge-balance-neg';
          ?>
            <tr>
              <td>
                <a href="finance/seller.php?username=<?= urlencode($r['username']) ?>">
                  <?= htmlspecialchars($r['username']) ?>
                </a>
              </td>
              <td><?= htmlspecialchars($r['full_name']) ?></td>
              <td><?= nf($r['orders_count']) ?></td>
              <td><?= nf($r['total_deposit']) ?></td>
              <td><?= nf($r['total_withdraw']) ?></td>
              <td class="<?= $cls ?>"><?= nf($remain) ?></td>
            </tr>
          <?php endforeach; ?>
          <?php if (empty($sellers)): ?>
            <tr><td colspan="6">هیچ فروشنده‌ای یافت نشد.</td></tr>
          <?php endif; ?>
          </tbody>
        </table>
      </div>
      <div class="pagination" id="sellerPagination"></div>
    </section>

  <?php else: ?>

    <!-- ================= آمار بالا برای فروشنده ================= -->
    <section class="card">
      <div class="card-header">
        <h3>💼 وضعیت حساب شما</h3>
      </div>
      <div class="stats-grid">
        <div class="stat-card">
          <div class="stat-label">جمع بیعانه‌های ثبت‌شده</div>
          <div class="stat-value"><?= nf($seller_total_deposit) ?> <span style="font-size:11px;">تومان</span></div>
          <div class="stat-chip">همه فاکتورهای شما</div>
        </div>
        <div class="stat-card">
          <div class="stat-label">جمع برداشت‌های ثبت‌شده</div>
          <div class="stat-value"><?= nf($seller_total_withdraw) ?> <span style="font-size:11px;">تومان</span></div>
          <div class="stat-chip">تسویه‌شده با شما</div>
        </div>
        <div class="stat-card">
          <div class="stat-label">مانده حساب فعلی</div>
          <div class="stat-value <?= $seller_balance >= 0 ? 'positive' : 'negative' ?>">
            <?= nf($seller_balance) ?> <span style="font-size:11px;">تومان</span>
          </div>
          <div class="stat-chip">بیعانه‌ها − برداشت‌ها</div>
        </div>
        <div class="stat-card">
          <div class="stat-label">ماه انتخاب‌شده</div>
          <div class="stat-value"><?= htmlspecialchars($jmonthParam) ?></div>
          <div class="stat-chip">تقویم کامل شمسی</div>
        </div>
      </div>
    </section>

    <!-- ================= جدول فاکتورهای ماه شمسی انتخاب‌شده ================= -->
    <section class="card">
      <div class="card-header">
        <h3>📄 فاکتورهای این ماه شمسی</h3>
      </div>

      <form method="get" class="table-toolbar">
        <div class="toolbar-group">
          <label for="jmonth" style="font-size:12px;color:#4b5563;">ماه شمسی:</label>
          <select id="jmonth" name="jmonth" class="select-sm">
            <?php foreach($monthOptions as $opt): ?>
              <option value="<?= htmlspecialchars($opt['value']) ?>"
                <?= $opt['value'] === $jmonthParam ? 'selected' : '' ?>>
                <?= htmlspecialchars($opt['label']) ?>
              </option>
            <?php endforeach; ?>
          </select>
          <button type="submit" class="btn-sm btn-primary">اعمال فیلتر</button>
        </div>
        <div class="toolbar-group">
          <input type="text" id="txSearch" class="input-sm" placeholder="🔍 جستجو در نام مشتری / توضیحات / فاکتور">
        </div>
      </form>

      <div class="table-wrapper">
        <table id="txTable">
          <thead>
            <tr>
              <th data-type="num">#</th>
              <th data-type="text">فاکتور</th>
              <th data-type="text">نام مشتری</th>
              <th data-type="text">تاریخ فاکتور (شمسی)</th>
              <th data-type="num">جمع فاکتور</th>
              <th data-type="num">بیعانه</th>
              <th data-type="num">COD</th>
              <th data-type="num">مانده</th>
              <th data-type="text">توضیحات</th>
              <th data-type="text">عملیات</th>
            </tr>
          </thead>
          <tbody>
          <?php
            $i = 1;
            foreach($transactions as $r):
          ?>
            <tr>
              <td><?= $i++ ?></td>
              <td><?= htmlspecialchars($r['invoice_no']) ?></td>
              <td><?= htmlspecialchars($r['customer_name']) ?></td>
              <td><?= toJalaliEn(substr($r['created_at'],0,10)) ?></td>
              <td><?= nf($r['total_amount']) ?></td>
              <td><?= nf($r['total_deposit']) ?></td>
              <td><?= nf($r['total_cod']) ?></td>
              <td><?= nf($r['total_remaining']) ?></td>
              <td>
                <?php if(trim($r['notes']) !== ''): ?>
                  <span class="tag-badge" title="<?= htmlspecialchars($r['notes']) ?>">دارای توضیح</span>
                <?php else: ?>
                  <span class="tag-badge">-</span>
                <?php endif; ?>
              </td>
              <td>
                <?php if(!empty($r['order_uid'])): ?>
                  <a href="../orders/view.php?id=<?= htmlspecialchars($r['order_uid']) ?>" class="btn-table primary" target="_blank">👁 مشاهده</a>
                  <a href="../orders/edit.php?id=<?= htmlspecialchars($r['order_uid']) ?>" class="btn-table ghost" target="_blank">✏️ ویرایش</a>
                <?php else: ?>
                  -
                <?php endif; ?>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (empty($transactions)): ?>
            <tr><td colspan="10">📭 در این ماه هیچ فاکتوری ثبت نشده است.</td></tr>
          <?php endif; ?>
          </tbody>
        </table>
      </div>
    </section>

  <?php endif; ?>

</div>

<script>
// ====================== نمودار Chart.js ======================
(function(){
  const labels   = <?= json_encode($chartLabels, JSON_UNESCAPED_UNICODE) ?>;
  const deposits = <?= json_encode($chartDeposit, JSON_UNESCAPED_UNICODE) ?>;
  const withdraws= <?= json_encode($chartWithdraw, JSON_UNESCAPED_UNICODE) ?>;
  const netVals  = <?= json_encode($chartNet, JSON_UNESCAPED_UNICODE) ?>;

  const canvas = document.getElementById('financeChart');
  const emptyMsg = document.getElementById('chartEmptyMsg');

  if (!canvas) return;

  if (!labels.length) {
    if (emptyMsg) emptyMsg.style.display = 'flex';
    return;
  } else {
    if (emptyMsg) emptyMsg.style.display = 'none';
  }

  const ctx = canvas.getContext('2d');
  new Chart(ctx, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'بیعانه',
          data: deposits,
          borderColor: '#0ea5e9',
          backgroundColor: 'rgba(14,165,233,0.15)',
          tension: 0.3,
          borderWidth: 2,
          pointRadius: 3
        },
        {
          label: 'برداشت',
          data: withdraws,
          borderColor: '#f97316',
          backgroundColor: 'rgba(249,115,22,0.15)',
          tension: 0.3,
          borderWidth: 2,
          pointRadius: 3
        },
        {
          label: 'خالص',
          data: netVals,
          borderColor: '#22c55e',
          backgroundColor: 'rgba(34,197,94,0.15)',
          tension: 0.3,
          borderWidth: 2,
          borderDash: [4,4],
          pointRadius: 3
        }
      ]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      interaction: {
        mode: 'index',
        intersect: false
      },
      plugins: {
        legend: {
          display: true,
          labels: {
            font: { size: 11, family: 'IRANSansWeb' }
          }
        },
        tooltip: {
          callbacks: {
            label: function(ctx){
              const v = ctx.parsed.y || 0;
              return ctx.dataset.label + ': ' + v.toLocaleString('en-US') + ' تومان';
            }
          }
        }
      },
      scales: {
        x: {
          ticks: { font:{ size:11, family:'IRANSansWeb' } }
        },
        y: {
          ticks: {
            font:{ size:11, family:'IRANSansWeb' },
            callback: function(val){return (val || 0).toLocaleString('en-US');}
          },
          beginAtZero: true
        }
      }
    }
  });
})();

// ====================== جدول فروشنده‌ها (admin) ======================
(function(){
  const table = document.getElementById('sellerTable');
  if (!table) return;

  const tbody = table.querySelector('tbody');
  const rows  = Array.from(tbody.querySelectorAll('tr'));
  const searchInput  = document.getElementById('sellerSearch');
  const balanceFilter = document.getElementById('balanceFilter');
  const paginationEl = document.getElementById('sellerPagination');

  const rowsPerPage = 15;
  let currentPage = 1;
  let sortIndex = -1;
  let sortDir   = 'asc';

  function parseNum(text){
    return parseFloat(text.replace(/,/g,'').trim()) || 0;
  }

  function getFilteredSortedRows(){
    let filtered = rows.filter(r => {
      const cells = r.cells;
      if (!cells.length) return false;

      const q = (searchInput.value || '').trim().toLowerCase();
      if (q){
        const hay = (cells[0].innerText + ' ' + cells[1].innerText).toLowerCase();
        if (!hay.includes(q)) return false;
      }

      const bVal = balanceFilter.value;
      if (bVal){
        const balanceText = cells[5].innerText || '0';
        const balance = parseNum(balanceText);
        if (bVal === 'pos' && !(balance > 0)) return false;
        if (bVal === 'neg' && !(balance < 0)) return false;
        if (bVal === 'zero' && balance !== 0) return false;
      }

      return true;
    });

    if (sortIndex >= 0){
      const th = table.querySelectorAll('th')[sortIndex];
      const type = th.dataset.type || 'text';
      filtered.sort((a,b)=>{
        let A = a.cells[sortIndex].innerText;
        let B = b.cells[sortIndex].innerText;
        if (type === 'num'){
          A = parseNum(A); B = parseNum(B);
        } else {
          A = A.toLowerCase(); B = B.toLowerCase();
        }
        if (A === B) return 0;
        if (sortDir === 'asc') return A > B ? 1 : -1;
        return A < B ? 1 : -1;
      });
    }

    return filtered;
  }

  function render(){
    const all = getFilteredSortedRows();
    const totalPages = Math.max(1, Math.ceil(all.length / rowsPerPage));
    if (currentPage > totalPages) currentPage = totalPages;

    const start = (currentPage - 1) * rowsPerPage;
    const pageRows = all.slice(start, start + rowsPerPage);

    tbody.innerHTML = '';
    pageRows.forEach(r => tbody.appendChild(r));

    paginationEl.innerHTML = '';
    for (let i=1;i<=totalPages;i++){
      const btn = document.createElement('button');
      btn.textContent = i;
      if (i === currentPage) btn.classList.add('active');
      btn.onclick = function(){
        currentPage = i;
        render();
      };
      paginationEl.appendChild(btn);
    }
  }

  table.querySelectorAll('th').forEach((th,idx)=>{
    th.addEventListener('click', ()=>{
      if (sortIndex === idx){
        sortDir = (sortDir === 'asc') ? 'desc' : 'asc';
      } else {
        sortIndex = idx;
        sortDir = 'asc';
      }
      table.querySelectorAll('th').forEach(x=>x.classList.remove('sort-asc','sort-desc'));
      th.classList.add(sortDir === 'asc' ? 'sort-asc' : 'sort-desc');
      render();
    });
  });

  if (searchInput)  searchInput.addEventListener('input', ()=>{currentPage=1;render();});
  if (balanceFilter) balanceFilter.addEventListener('change', ()=>{currentPage=1;render();});

  render();
})();

// ====================== جستجو در جدول تراکنش‌های فروشنده ======================
(function(){
  const table = document.getElementById('txTable');
  const searchInput = document.getElementById('txSearch');
  if (!table || !searchInput) return;

  const tbody = table.querySelector('tbody');
  const rows  = Array.from(tbody.querySelectorAll('tr'));

  searchInput.addEventListener('input', function(){
    const q = this.value.trim().toLowerCase();
    rows.forEach(r=>{
      const t = r.innerText.toLowerCase();
      r.style.display = (!q || t.includes(q)) ? '' : 'none';
    });
  });
})();
</script>
</body>
</html>
