<?php
// ==============================
// Debug safe
// ==============================
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__.'/includes/auth.php';
require_once __DIR__.'/includes/config.php';
require_once __DIR__.'/includes/db_connect.php';
require_once __DIR__.'/includes/jdf.php';

// ------------------------------
// Helpers
// ------------------------------
function nf($n){ return number_format((int)$n, 0, '.', ','); }
function pad2($v){ return str_pad((string)($v ?? ''), 2, '0', STR_PAD_LEFT); }
function toJalaliEn($date){
  if(!$date || !preg_match('/^\d{4}-\d{2}-\d{2}/',$date)) return '-';
  list($gy,$gm,$gd) = explode('-', substr($date,0,10));
  $j = gregorian_to_jalali((int)$gy,(int)$gm,(int)$gd,'/');
  return strtr($j, ['۰'=>'0','۱'=>'1','۲'=>'2','۳'=>'3','۴'=>'4','۵'=>'5','۶'=>'6','۷'=>'7','۸'=>'8','۹'=>'9']);
}
function ym_to_jalali_label($ym){ // "YYYY-MM" -> "YYYY/MM" (Jalali)
  $parts = explode('-', $ym);
  if(count($parts) < 2) return $ym;
  $gy=(int)$parts[0]; $gm=(int)$parts[1];
  // تبدیل روز 1 همان ماه
  list($jy,$jm,$jd) = gregorian_to_jalali($gy,$gm,1);
  return $jy.'/'.pad2($jm);
}

// ------------------------------
// Session / Role
// ------------------------------
$user = $_SESSION['username'];
$role = $_SESSION['role'] ?? '';
$is_admin = ($role === 'admin');

// ------------------------------
// Filters
// ------------------------------
$brand  = trim($_GET['brand'] ?? '');
$month  = (int)($_GET['month'] ?? 0);
$year   = (int)($_GET['year'] ?? 0);
$seller = $is_admin ? trim($_GET['seller'] ?? '') : $user;
$export = trim($_GET['export'] ?? '');

// ------------------------------
// WHERE builder (+ optional date range)
// ------------------------------
$where = "1";
if($brand){
  $term = $mysqli->real_escape_string($brand);
  $where .= " AND (LOWER(oi.model) LIKE LOWER('%$term%'))";
}

// شمسی -> میلادی
$dateStart = null; $dateEnd = null;
if($year > 0 || $month > 0){
  $jy = $year ?: jdate('Y');
  $jm = $month ?: 1;

  list($gsY,$gsM,$gsD) = jalali_to_gregorian($jy,$jm,1,'-');
  $gsM = pad2($gsM); $gsD = pad2($gsD);

  if($month > 0){
    $nextY = ($month==12)?$jy+1:$jy;
    $nextM = ($month==12)?1:$month+1;
    list($geY,$geM,$geD) = jalali_to_gregorian($nextY,$nextM,1,'-');
  } else {
    list($geY,$geM,$geD) = jalali_to_gregorian($jy+1,1,1,'-');
  }
  $geM = pad2($geM); $geD = pad2($geD);

  $dateStart = "{$gsY}-{$gsM}-{$gsD}";
  $dateEnd   = "{$geY}-{$geM}-{$geD}";
  $where    .= " AND o.purchase_date >= '{$dateStart}' AND o.purchase_date < '{$dateEnd}'";
}

// فروشنده
if(!$is_admin){
  $where .= " AND o.username='".$mysqli->real_escape_string($seller)."'";
} elseif($seller){
  $where .= " AND o.username='".$mysqli->real_escape_string($seller)."'";
}

// ------------------------------
// Base rows (table)
// ------------------------------
$sql_rows = "
SELECT oi.model, oi.price, o.username, o.purchase_date, o.id AS order_id
FROM order_items oi
JOIN orders o ON o.id = oi.order_id
WHERE $where
ORDER BY o.purchase_date DESC
";
$res_rows = $mysqli->query($sql_rows);
$rows = $res_rows->fetch_all(MYSQLI_ASSOC);

// ------------------------------
// Aggregates
// ------------------------------
$total_items = count($rows);
$total_value = 0;
$brand_summary = [];
$seller_summary = [];
$order_ids = [];

foreach($rows as $r){
  $price = (int)$r['price'];
  $total_value += $price;
  $order_ids[$r['order_id']] = true;

  // برند تخمینی از متن مدل
  $m = strtoupper(preg_replace('/[^A-Za-z0-9آ-ی]/','',$r['model']));
  if(stripos($m,'LG') !== false) $brandName='LG';
  elseif(stripos($m,'SAMSUNG') !== false) $brandName='SAMSUNG';
  elseif(stripos($m,'SONY') !== false) $brandName='SONY';
  elseif(stripos($m,'TCL') !== false) $brandName='TCL';
  elseif(stripos($m,'XIAOMI') !== false) $brandName='XIAOMI';
  else $brandName='سایر برندها';

  $brand_summary[$brandName] = ($brand_summary[$brandName] ?? 0) + $price;
  $seller_summary[$r['username']] = ($seller_summary[$r['username']] ?? 0) + $price;
}

$total_orders = count($order_ids);
$avg_per_order = ($total_orders>0) ? (int)round($total_value / $total_orders) : 0;

// ------------------------------
// Seller pivot (total & count items)
// ------------------------------
$sql_seller_pivot = "
SELECT o.username, SUM(oi.price) AS total, COUNT(*) AS items
FROM order_items oi
JOIN orders o ON o.id = oi.order_id
WHERE $where
GROUP BY o.username
ORDER BY total DESC
";
$res_seller_pivot = $mysqli->query($sql_seller_pivot);
$seller_pivot = $res_seller_pivot->fetch_all(MYSQLI_ASSOC);

// ------------------------------
// Top models (Top-10)
// ------------------------------
$sql_top_models = "
SELECT oi.model, SUM(oi.price) AS total, COUNT(*) AS items
FROM order_items oi
JOIN orders o ON o.id = oi.order_id
WHERE $where
GROUP BY oi.model
ORDER BY total DESC
LIMIT 10
";
$res_top = $mysqli->query($sql_top_models);
$top_models = $res_top->fetch_all(MYSQLI_ASSOC);

// ------------------------------
// Monthly trend (last 6 months if no date filter)
// ------------------------------
$trend_where = $where;
if(!$dateStart || !$dateEnd){
  // از 6 ماه قبل تا ابتدای ماه بعد
  $start = date('Y-m-01', strtotime('-5 months')); // 6 ماه شامل ماه جاری
  $end   = date('Y-m-01', strtotime('+1 month', strtotime(date('Y-m-01'))));
  $trend_where = "1";
  if($brand){
    $term = $mysqli->real_escape_string($brand);
    $trend_where .= " AND (LOWER(oi.model) LIKE LOWER('%$term%'))";
  }
  if(!$is_admin){
    $trend_where .= " AND o.username='".$mysqli->real_escape_string($seller)."'";
  } elseif($seller){
    $trend_where .= " AND o.username='".$mysqli->real_escape_string($seller)."'";
  }
  $trend_where .= " AND o.purchase_date >= '{$start}' AND o.purchase_date < '{$end}'";
}

$sql_trend = "
SELECT DATE_FORMAT(o.purchase_date, '%Y-%m') AS ym, SUM(oi.price) AS total
FROM order_items oi
JOIN orders o ON o.id = oi.order_id
WHERE $trend_where
GROUP BY ym
ORDER BY ym ASC
";
$res_trend = $mysqli->query($sql_trend);
$trend_rows = $res_trend->fetch_all(MYSQLI_ASSOC);

// ------------------------------
// Export CSV (rows)
// ------------------------------
if($export === 'csv'){
  header('Content-Type: text/csv; charset=UTF-8');
  header('Content-Disposition: attachment; filename="report.csv"');
  $out = fopen('php://output','w');
  // BOM برای اکسل
  fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));
  fputcsv($out, ['Seller','Model','Price','Purchase Date (Jalali)']);
  foreach($rows as $r){
    fputcsv($out, [
      $r['username'],
      $r['model'],
      (int)$r['price'],
      toJalaliEn($r['purchase_date'])
    ]);
  }
  fclose($out);
  exit;
}

// ------------------------------
// Sellers list (admin)
// ------------------------------
$sellers = [];
if($is_admin){
  $q=$mysqli->query("SELECT username, full_name FROM users WHERE role='seller' ORDER BY username");
  $sellers=$q->fetch_all(MYSQLI_ASSOC);
}

// ------------------------------
// Page render
// ------------------------------
?>
<!doctype html>
<html lang="fa" dir="rtl">
<meta charset="utf-8">
<title>📈 گزارش مدیریتی فروش | سلام‌بابا</title>
<link rel="stylesheet" href="assets/reports.css">
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>

<body>
<div class="container">
  <?php require __DIR__.'/includes/header.php'; ?>

  <section class="card report-card">
   <h3>
  📊 گزارش مدیریتی فروش (شمسی)
  <button class="toggle-theme" onclick="toggleTheme()">🌓 حالت تاریک</button>
</h3>


    <form class="filters" method="get">
      <input type="text" name="brand" placeholder="🔍 برند یا مدل (مثلاً LG)" value="<?=htmlspecialchars($brand)?>">
      <select name="month">
        <option value="0" <?=$month==0?'selected':''?>>📅 همه ماه‌ها</option>
        <?php
        $months=['فروردین','اردیبهشت','خرداد','تیر','مرداد','شهریور','مهر','آبان','آذر','دی','بهمن','اسفند'];
        for($m=1;$m<=12;$m++): ?>
          <option value="<?=$m?>" <?=$m==$month?'selected':''?>><?=$months[$m-1]?></option>
        <?php endfor; ?>
      </select>

      <select name="year">
        <option value="0" <?=$year==0?'selected':''?>>📆 همه سال‌ها</option>
        <?php for($y=1400;$y<=1405;$y++): ?>
          <option value="<?=$y?>" <?=$y==$year?'selected':''?>><?=$y?></option>
        <?php endfor; ?>
      </select>

      <?php if($is_admin): ?>
      <select name="seller">
        <option value="">👥 همه فروشنده‌ها</option>
        <?php foreach($sellers as $s): ?>
          <option value="<?=$s['username']?>" <?=$seller==$s['username']?'selected':''?>><?=$s['full_name']?> (<?=$s['username']?>)</option>
        <?php endforeach; ?>
      </select>
      <?php endif; ?>

      <button class="btn primary">نمایش گزارش</button>
      <a class="btn ghost" href="?<?=http_build_query(array_merge($_GET,['export'=>'csv']))?>">📤 خروجی CSV</a>
    </form>

    <!-- KPI Bar -->
    <div class="kpis">
      <div class="kpi">
        <div class="kpi-title">💰 مبلغ کل</div>
        <div class="kpi-value"><?=nf($total_value)?> <span>تومان</span></div>
      </div>
      <div class="kpi">
        <div class="kpi-title">🛒 تعداد آیتم</div>
        <div class="kpi-value"><?=nf($total_items)?></div>
      </div>
      <div class="kpi">
        <div class="kpi-title">🧾 تعداد سفارش</div>
        <div class="kpi-value"><?=nf($total_orders)?></div>
      </div>
      <div class="kpi">
        <div class="kpi-title">📉 میانگین هر سفارش</div>
        <div class="kpi-value"><?=nf($avg_per_order)?> <span>تومان</span></div>
      </div>
      <?php if($brand): ?>
      <div class="kpi chip">🏷 جستجو: <?=htmlspecialchars($brand)?></div>
      <?php endif; ?>
      <?php if($dateStart && $dateEnd): ?>
      <div class="kpi chip">⏱ بازه: <?=toJalaliEn($dateStart)?> تا <?=toJalaliEn(date('Y-m-d', strtotime($dateEnd.' -1 day')))?></div>
      <?php endif; ?>
    </div>

    <!-- Charts -->
    <div class="charts">
      <div class="chartBox">
        <h4>📊 فروش بر اساس فروشنده</h4>
        <canvas id="sellerChart"></canvas>
      </div>
      <div class="chartBox">
        <h4>🥧 سهم برندها از کل فروش</h4>
        <canvas id="brandChart"></canvas>
      </div>
      <div class="chartBox">
        <h4>📈 روند ماه‌به‌ماه</h4>
        <canvas id="trendChart"></canvas>
      </div>
      <div class="chartBox">
        <h4>🏆 ۱۰ مدل پرفروش</h4>
        <canvas id="topModelsChart"></canvas>
      </div>
    </div>

    <!-- Seller pivot table -->
    <div class="table-wrapper">
      <h4 class="table-title">گزارش فروشندگان (بازهٔ فعلی)</h4>
      <table>
        <thead>
          <tr>
            <th>فروشنده</th>
            <th>تعداد آیتم</th>
            <th>مجموع فروش (تومان)</th>
            <th>میانگین هر آیتم (تومان)</th>
          </tr>
        </thead>
        <tbody>
          <?php if(!$seller_pivot): ?>
            <tr><td colspan="4">داده‌ای یافت نشد.</td></tr>
          <?php else: foreach($seller_pivot as $sp): 
              $avg = ($sp['items']>0)? (int)round($sp['total']/$sp['items']) : 0; ?>
            <tr>
              <td><?=htmlspecialchars($sp['username'])?></td>
              <td><?=nf($sp['items'])?></td>
              <td><?=nf($sp['total'])?></td>
              <td><?=nf($avg)?></td>
            </tr>
          <?php endforeach; endif; ?>
        </tbody>
      </table>
    </div>

    <!-- Raw rows table -->
    <div class="table-wrapper">
      <h4 class="table-title">جزئیات آیتم‌ها</h4>
      <table>
        <thead>
          <tr>
            <?php if($is_admin): ?><th>فروشنده</th><?php endif; ?>
            <th>مدل</th>
            <th>قیمت</th>
            <th>تاریخ خرید (شمسی)</th>
          </tr>
        </thead>
        <tbody>
          <?php if(!$rows): ?>
            <tr><td colspan="<?=$is_admin?4:3?>">هیچ آیتمی یافت نشد.</td></tr>
          <?php else: foreach($rows as $r): ?>
            <tr>
              <?php if($is_admin): ?><td><?=htmlspecialchars($r['username'])?></td><?php endif; ?>
              <td><?=htmlspecialchars($r['model'])?></td>
              <td><?=nf($r['price'])?></td>
              <td><?=toJalaliEn($r['purchase_date'])?></td>
            </tr>
          <?php endforeach; endif; ?>
        </tbody>
      </table>
    </div>

  </section>
</div>

<script>
Chart.defaults.maintainAspectRatio = false;
Chart.defaults.responsive = true;

const sellerData  = <?=json_encode($seller_summary, JSON_UNESCAPED_UNICODE)?>;
const brandData   = <?=json_encode($brand_summary,  JSON_UNESCAPED_UNICODE)?>;
const trendRows   = <?=json_encode($trend_rows,     JSON_UNESCAPED_UNICODE)?>;
const topModels   = <?=json_encode($top_models,     JSON_UNESCAPED_UNICODE)?>;

function jLabel(ym){ // "YYYY-MM" -> Jalali "YYYY/MM"
  return ym ? "<?= '' ?>" + ym : ym;
}

if(Object.keys(sellerData).length){
  new Chart(document.getElementById('sellerChart'), {
    type: 'bar',
    data: {
      labels: Object.keys(sellerData),
      datasets: [{ label: 'مبلغ فروش (تومان)', data: Object.values(sellerData),
        backgroundColor: 'rgba(37,99,235,0.7)', borderRadius: 6 }]
    },
    options: { scales:{y:{beginAtZero:true}}, plugins:{legend:{display:false}} }
  });
}

if(Object.keys(brandData).length){
  new Chart(document.getElementById('brandChart'), {
    type: 'pie',
    data: {
      labels: Object.keys(brandData),
      datasets: [{ data: Object.values(brandData),
        backgroundColor:['#2563eb','#16a34a','#f59e0b','#dc2626','#7c3aed','#9ca3af','#10b981','#ef4444'] }]
    },
    options: { plugins:{legend:{position:'bottom'}} }
  });
}

if(Array.isArray(trendRows) && trendRows.length){
  const tLabels = trendRows.map(r => r.ym); // برچسب میلادی
  const tData   = trendRows.map(r => parseInt(r.total||0));
  new Chart(document.getElementById('trendChart'), {
    type:'line',
    data:{ labels: tLabels, datasets:[{ label:'جمع فروش ماهانه', data:tData, tension:0.3 }] },
    options:{ plugins:{legend:{display:false}}, scales:{y:{beginAtZero:true}} }
  });
}

if(Array.isArray(topModels) && topModels.length){
  new Chart(document.getElementById('topModelsChart'), {
    type: 'bar',
    data: {
      labels: topModels.map(r => r.model),
      datasets: [{ label:'مبلغ فروش', data: topModels.map(r=>parseInt(r.total||0)),
        backgroundColor:'rgba(99,102,241,0.7)', borderRadius:6 }]
    },
    options: {
      indexAxis:'y',
      plugins:{legend:{display:false}},
      scales:{x:{beginAtZero:true}}
    }
  });
}
</script>
<script>
function toggleTheme(){
  document.body.classList.toggle('dark');
  localStorage.setItem('theme', document.body.classList.contains('dark') ? 'dark' : 'light');
}
if(localStorage.getItem('theme')==='dark') document.body.classList.add('dark');
</script>

</body>
</html>
