<?php
// delivery.php - فرم تحویل برای راننده با لینک رمزدار

require __DIR__ . '/includes/config.php';
require __DIR__ . '/includes/db_connect.php';
require __DIR__ . '/includes/jdf.php';

if (!defined('DELIVERY_LINK_SECRET')) {
    define('DELIVERY_LINK_SECRET', 'SLMB_DRIVE_2025_SECRET_KEY');
}

// ساخت توکن امن
function make_delivery_token($order_uid) {
    return hash_hmac('sha256', $order_uid, DELIVERY_LINK_SECRET);
}

function e($s) {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}

$status_map = [
    'new'               => '🕓 ثبت شده (در انتظار بررسی)',
    'assigned_to_buyer' => '📌 ارجاع به مسئول خرید',
    'purchased'         => '🛒 خریداری شده',
    'handover_to_head'  => '📦 تحویل به سر باربر',
    'out_for_delivery'  => '🚚 در حال ارسال',
    'delivered'         => '✅ تحویل داده شد',
    'failed'            => '❌ تحویل ناموفق'
];

// ------------------------------------------------------
// دریافت ورودی لینک رمزدار
// ------------------------------------------------------
$code  = trim($_GET['code'] ?? '');
$token = trim($_GET['token'] ?? '');

$order = null;
$order_items = [];
$errors = [];
$success = "";

// ------------------------------------------------------
// 1) بررسی لینک ورودی
// ------------------------------------------------------
if ($code === '' || $token === '') {
    $errors[] = "لینک ناقص است.";
} else {

    $st = $mysqli->prepare("SELECT * FROM orders WHERE tracking_code=? LIMIT 1");
    $st->bind_param("s", $code);
    $st->execute();
    $order = $st->get_result()->fetch_assoc();
    $st->close();

    if (!$order) {
        $errors[] = "سفارش یافت نشد.";
    } else {

        $expected = make_delivery_token($order['order_uid']);

        if (!hash_equals($expected, $token)) {
            $errors[] = "لینک تحویل معتبر نیست.";
            $order = null;
        }
    }
}

// ------------------------------------------------------
// 2) اگر سفارش معتبر بود → اقلام را هم بگیر
// ------------------------------------------------------
if ($order) {

    $oid = (int)$order['id'];

    $st = $mysqli->prepare("SELECT * FROM order_items WHERE order_id=?");
    $st->bind_param("i", $oid);
    $st->execute();
    $r = $st->get_result();
    while ($x = $r->fetch_assoc()) $order_items[] = $x;
    $st->close();
}

// ------------------------------------------------------
// 3) اگر POST شد → ثبت تحویل
// ------------------------------------------------------
if ($order && $_SERVER['REQUEST_METHOD'] === 'POST') {

    $delivery_status = $_POST['delivery_status'] ?? '';
    $driver_name     = trim($_POST['driver_name'] ?? '');
    $serial_number   = trim($_POST['serial_number'] ?? '');
    $customer_name   = trim($_POST['customer_name'] ?? '');
    $customer_phone  = trim($_POST['customer_phone'] ?? '');
    $address_snap    = trim($_POST['address_snapshot'] ?? '');
    $note            = trim($_POST['note'] ?? '');

    if (!in_array($delivery_status, ['delivered','failed'])) {
        $errors[] = "وضعیت تحویل را انتخاب کنید.";
    }

    // مسیرهای ذخیره‌سازی
    $base = __DIR__ . "/uploads/delivery";
    $dir_photos = $base . "/photos";
    $dir_media  = $base . "/media";

    foreach([$base,$dir_photos,$dir_media] as $d){
        if(!is_dir($d)) @mkdir($d,0775,true);
    }

    $photo1 = $photo2 = $photo3 = null;
    $video1 = null;

    $hasFile = false;

    // آپلود عکس‌ها
    foreach (['photo1','photo2','photo3'] as $field) {

        if (!empty($_FILES[$field]['name']) && $_FILES[$field]['error'] === UPLOAD_ERR_OK) {

            $ext = strtolower(pathinfo($_FILES[$field]['name'], PATHINFO_EXTENSION));
            if (!in_array($ext, ['jpg','jpeg','png','webp'])) {
                $errors[] = "فرمت عکس {$field} مجاز نیست.";
                continue;
            }

            $new = "o{$order['id']}_".time()."_{$field}.{$ext}";
            $dest = $dir_photos . "/" . $new;

            if (move_uploaded_file($_FILES[$field]['tmp_name'], $dest)) {
                $hasFile = true;

                if ($field === 'photo1') $photo1 = "uploads/delivery/photos/".$new;
                if ($field === 'photo2') $photo2 = "uploads/delivery/photos/".$new;
                if ($field === 'photo3') $photo3 = "uploads/delivery/photos/".$new;
            }
        }
    }

    // ویدیو / صوت
    if (!empty($_FILES['media1']['name']) && $_FILES['media1']['error'] === UPLOAD_ERR_OK) {

        $ext = strtolower(pathinfo($_FILES['media1']['name'], PATHINFO_EXTENSION));
        $allowed_media = ['mp4','mov','mkv','3gp','mp3','ogg','m4a','wav'];

        if (!in_array($ext, $allowed_media)) {
            $errors[] = "فرمت فایل ویدیو/صوت مجاز نیست.";
        } else {
            $new = "o{$order['id']}_".time()."_media.{$ext}";
            $dest = $dir_media . "/" . $new;

            if (move_uploaded_file($_FILES['media1']['tmp_name'], $dest)) {
                $hasFile = true;
                $video1 = "uploads/delivery/media/".$new;
            }
        }
    }

    // اگر تحویل موفق → حداقل یک مدرک لازم است
    if ($delivery_status === "delivered" && !$hasFile) {
        $errors[] = "برای تحویل موفق حداقل یک عکس یا ویدیو لازم است.";
    }

    if (empty($errors)) {

        // ثبت در جدول مدارک تحویل
        $chk = $mysqli->query("SHOW TABLES LIKE 'order_delivery_proofs'");
        if (!$chk || $chk->num_rows === 0) {
            $errors[] = "جدول مدارک تحویل موجود نیست.";
        } else {

            $item_id = null;

            $st = $mysqli->prepare("
                INSERT INTO order_delivery_proofs
                (order_id,item_id,serial_number,customer_name,customer_phone,
                 address_snapshot,note,photo1,photo2,photo3,video1,created_by)
                VALUES (?,?,?,?,?,?,?,?,?,?,?,?)
            ");

            $st->bind_param(
                "iissssssssss",
                $order['id'],
                $item_id,
                $serial_number,
                $customer_name,
                $customer_phone,
                $address_snap,
                $note,
                $photo1,
                $photo2,
                $photo3,
                $video1,
                $driver_name
            );

            if ($st->execute()) {

                // آپدیت وضعیت سفارش
                $new_status = ($delivery_status === 'delivered') ? 'delivered' : 'failed';
                $extra_note = ($new_status === 'delivered')
                    ? "\nتحویل راننده: تحویل موفق ثبت شد."
                    : "\nتحویل راننده: تحویل ناموفق ثبت شد.";

                $st2 = $mysqli->prepare("
                    UPDATE orders
                    SET shipping_status=?, status_note = CONCAT(COALESCE(status_note,''), ?)
                    WHERE id=?
                ");
                $st2->bind_param("ssi", $new_status, $extra_note, $order['id']);
                $st2->execute();
                $st2->close();

                $success = "اطلاعات تحویل با موفقیت ثبت شد.";
            } else {
                $errors[] = "خطا در ثبت اطلاعات.";
            }

            $st->close();
        }
    }
}
?>
<!doctype html>
<html lang="fa" dir="rtl">
<head>
<meta charset="utf-8">
<title>فرم تحویل سفارش | سلام بابا</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<style>
body{font-family:tahoma;background:#f5f5f5;margin:0;}
.container{max-width:700px;margin:20px auto;background:#fff;padding:20px;border-radius:8px}
h1{text-align:center;font-size:18px;margin:0 0 10px}
.msg-error{background:#ffecec;color:#b30000;padding:8px;border-radius:4px}
.msg-success{background:#e8ffe8;color:#007700;padding:8px;border-radius:4px}
.order-box{background:#fafafa;padding:10px;border:1px solid #ddd;border-radius:6px;margin-bottom:15px;font-size:13px}
input,textarea,select{width:100%;padding:6px;margin-top:4px;border:1px solid #ccc;border-radius:4px;font-size:13px}
button{width:100%;padding:10px;background:#28a745;color:#fff;border:none;border-radius:4px;font-size:15px;margin-top:10px}
</style>
</head>
<body>

<div class="container">
    <h1>فرم تحویل سفارش</h1>

    <?php if ($errors): ?>
        <div class="msg-error">
        <?php foreach($errors as $er) echo "<div>".e($er)."</div>"; ?>
        </div>
    <?php endif; ?>

    <?php if ($success): ?>
        <div class="msg-success"><?php echo e($success); ?></div>
    <?php endif; ?>

    <?php if ($order): ?>

        <div class="order-box">
            <div><b>کد پیگیری:</b> <?php echo e($order['tracking_code']); ?></div>
            <div><b>نام مشتری:</b> <?php echo e($order['customer_name']); ?></div>
            <div><b>موبایل:</b> <?php echo e($order['mobile1']); ?></div>
            <div><b>آدرس:</b> <?php echo e($order['address']); ?></div>
            <div><b>وضعیت فعلی:</b> <?php echo e($status_map[$order['shipping_status']] ?? $order['shipping_status']); ?></div>
        </div>

        <form method="post" enctype="multipart/form-data">

            <label>وضعیت تحویل</label>
            <select name="delivery_status">
                <option value="">انتخاب کنید...</option>
                <option value="delivered">تحویل موفق به مشتری</option>
                <option value="failed">تحویل ناموفق</option>
            </select>

            <label>نام راننده</label>
            <input type="text" name="driver_name">

            <label>سریال دستگاه</label>
            <input type="text" name="serial_number">

            <label>نام مشتری</label>
            <input type="text" name="customer_name" value="<?php echo e($order['customer_name']); ?>">

            <label>شماره تماس مشتری</label>
            <input type="text" name="customer_phone" value="<?php echo e($order['mobile1']); ?>">

            <label>توضیح کوتاه از محل/آدرس</label>
            <textarea name="address_snapshot"></textarea>

            <label>توضیحات راننده</label>
            <textarea name="note"></textarea>

            <hr>

            <label>عکس محصول (تا ۳ عدد)</label>
            <input type="file" name="photo1">
            <input type="file" name="photo2">
            <input type="file" name="photo3">

            <label>فایل ویدیو/صوت (اختیاری)</label>
            <input type="file" name="media1">

            <button type="submit">ثبت اطلاعات تحویل</button>

        </form>

    <?php else: ?>
        <p>اطلاعات سفارش قابل نمایش نیست.</p>
    <?php endif; ?>
</div>

</body>
</html>
