<?php
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

require __DIR__.'/includes/config.php';
require __DIR__.'/includes/db_connect.php';
require __DIR__.'/includes/jdf.php';

$msg = "";
$msg_type = ""; // success | error

// تابع کمکی برای تمیز کردن ورودی
function clean($s){
    return trim($s ?? '');
}

// تابع تبدیل تاریخ شمسی فرم به میلادی برای دیتابیس
function jalaliFormToGregorianDate($jalali){
    $jalali = trim($jalali);
    if(!$jalali || !preg_match('/^\d{4}-\d{2}-\d{2}$/',$jalali)){
        return null;
    }
    list($jy,$jm,$jd) = explode('-',$jalali);
    $g = jalali_to_gregorian((int)$jy,(int)$jm,(int)$jd,'-');
    return $g; // yyyy-mm-dd
}

// تابع آپلود فایل با چک حداقلی
function uploadFile($key,$uploadDir){
    if(!isset($_FILES[$key]) || $_FILES[$key]['error'] !== UPLOAD_ERR_OK){
        return "";
    }

    $allowedExt = ['jpg','jpeg','png','gif','pdf'];
    $maxSize    = 5 * 1024 * 1024; // 5MB

    $file = $_FILES[$key];

    if($file['size'] > $maxSize){
        return "";
    }

    $ext = strtolower(pathinfo($file['name'],PATHINFO_EXTENSION));
    if(!in_array($ext,$allowedExt)){
        return "";
    }

    if(!is_dir($uploadDir)){
        mkdir($uploadDir,0777,true);
    }

    $name = uniqid('f_',true).".".$ext;
    if(move_uploaded_file($file['tmp_name'],$uploadDir.$name)){
        return $name;
    }
    return "";
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $role        = clean($_POST['role']);
    $full_name   = clean($_POST['full_name']);
    $mobile      = clean($_POST['mobile']);
    $username    = clean($_POST['username']); // نام کاربری پیشنهادی
    $national_id = clean($_POST['national_id']);
    $birth_raw   = clean($_POST['birthdate']); // شمسی از فرم
    $address     = clean($_POST['address']);
    $vehicle     = clean($_POST['vehicle']);
    $plate       = clean($_POST['plate']);
    $password    = clean($_POST['password']);

    // اگر نام کاربری خالی بود از موبایل استفاده کن
    if ($username === '' && $mobile !== '') {
        $username = $mobile;
    }

    // ولیدیشن های حداقلی
    $errors = [];

    $allowedRoles = ['seller','buyer','logistic','head_carrier','support','admin'];
    if(!in_array($role,$allowedRoles)){
        $errors[] = "نقش کاربر نامعتبر است.";
    }

    if($full_name === ''){
        $errors[] = "نام و نام خانوادگی الزامی است.";
    }

    if(!preg_match('/^09\d{9}$/',$mobile)){
        $errors[] = "شماره موبایل باید با 09 شروع شود و 11 رقم باشد.";
    }

    if(strlen($username) < 4){
        $errors[] = "نام کاربری باید حداقل 4 کاراکتر باشد.";
    }

    if(strlen($password) < 6){
        $errors[] = "رمز عبور باید حداقل 6 کاراکتر باشد.";
    }

    // تبدیل تاریخ شمسی به میلادی برای ذخیره
    $birthdate = jalaliFormToGregorianDate($birth_raw);

    if(empty($errors)){
        // چک تکراری بودن نام کاربری یا موبایل
        $stmt = $mysqli->prepare("SELECT id FROM users WHERE username=? OR mobile=? LIMIT 1");
        if($stmt){
            $stmt->bind_param("ss",$username,$mobile);
            $stmt->execute();
            $stmt->store_result();
            if($stmt->num_rows > 0){
                $errors[] = "نام کاربری یا شماره موبایل قبلا ثبت شده است.";
            }
            $stmt->close();
        } else {
            $errors[] = "خطای دیتابیس در بررسی تکراری بودن.";
        }
    }

    // اگر خطا نداریم، بریم برای درج
    if(empty($errors)){

        $uploadDir = __DIR__.'/uploads/logistic/';

        $photo_face = uploadFile("photo_face",$uploadDir);
        $photo_nid  = uploadFile("photo_nid",$uploadDir);
        $photo_car  = uploadFile("photo_car",$uploadDir);

        $password_hash = password_hash($password,PASSWORD_BCRYPT);

        $sql = "
            INSERT INTO users
            (username,password_hash,full_name,role,mobile,national_id,birthdate,address,vehicle,plate,photo_face,photo_nid,photo_car,is_approved)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,0)
        ";
        $stmt = $mysqli->prepare($sql);
        if(!$stmt){
            $msg = "خطای دیتابیس در آماده سازی کوئری.";
            $msg_type = "error";
        } else {
            $stmt->bind_param(
                "sssssssssssss",
                $username,
                $password_hash,
                $full_name,
                $role,
                $mobile,
                $national_id,
                $birthdate,
                $address,
                $vehicle,
                $plate,
                $photo_face,
                $photo_nid,
                $photo_car
            );

            if($stmt->execute()){
                $msg = "ثبت نام انجام شد. منتظر تاييد مديريت باشيد.";
                $msg_type = "success";
                // خالی کردن فرم بعد از موفقیت
                $role = $full_name = $mobile = $username = $national_id = $birth_raw = $address = $vehicle = $plate = "";
            } else {
                $msg = "خطا در ثبت نام: ".$stmt->error;
                $msg_type = "error";
            }
            $stmt->close();
        }
    } else {
        $msg = implode("<br>",$errors);
        $msg_type = "error";
    }
}
?>
<!doctype html>
<html lang="fa" dir="rtl">

<head>
<meta charset="utf-8">
<title>ثبت نام - سلام بابا</title>
<meta name="viewport" content="width=device-width, initial-scale=1">

<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/persian-datepicker@1.2.0/dist/css/persian-datepicker.min.css">

<style>
/* ==========================================
   Font Setup (Vazirmatn + IRANSansWeb)
========================================== */
@font-face {
  font-family: 'Vazirmatn';
  src: url('https://salambaba.co/orders/font/Vazirmatn-Regular.woff2') format('woff2');
  font-weight: 400;
  font-display: swap;
}
@font-face {
  font-family: 'Vazirmatn';
  src: url('https://salambaba.co/orders/font/Vazirmatn-Bold.woff2') format('woff2');
  font-weight: 700;
  font-display: swap;
}
@font-face {
  font-family: 'IRANSansWeb';
  src: url('https://salambaba.co/font/IRANSansWeb.woff2') format('woff2'),
       url('https://salambaba.co/font/IRANSansWeb.woff') format('woff');
  font-weight: 400;
  font-display: swap;
}
@font-face {
  font-family: 'IRANSansWeb';
  src: url('https://salambaba.co/font/IRANSansWeb_Bold.woff2') format('woff2'),
       url('https://salambaba.co/font/IRANSansWeb_Bold.woff') format('woff');
  font-weight: 700;
  font-display: swap;
}

/* ==========================================
   Theme Variables
========================================== */
:root {
  --brand-blue: #007BFF;
  --brand-green: #22c55e;
  --brand-yellow: #facc15;
  --brand-red: #ef4444;
  --text: #0f172a;
  --muted: #6c757d;
  --card: #ffffff;
  --shadow: 0 6px 18px rgba(15,23,42,.08);
  --border: #d1d5db;
}

/* ==========================================
   Global Layout
========================================== */
html, body {
  font-family: 'Vazirmatn', 'IRANSansWeb', Tahoma, Arial, sans-serif;
  direction: rtl;
  text-align: right;
  background: linear-gradient(180deg, #f9fafb 0%, #f1f5f9 100%);
  color: var(--text);
  min-height: 100vh;
  display: flex;
  justify-content: center;
  align-items: center;
  padding: 3px;
  margin: 0;
}

/* ==========================================
   Logo Stripes
========================================== */
.logo-stripes {
  display: flex;
  gap: 6px;
  align-items: center;
  justify-content: center;
  background: var(--card);
  padding: 19px 0;
  border-radius: 18px;
  box-shadow: var(--shadow);
  animation: blink 3.5s infinite alternate;
  margin-bottom: 22px;
}
.logo-stripes span {
  display: block;
  width: 20px;
  height: 48px;
  border-radius: 20px;
  animation: wave 5s infinite ease-in-out;
}
.b { background: var(--brand-blue); animation-delay: 0s; }
.g { background: var(--brand-green); animation-delay: 0.1s; }
.y { background: var(--brand-yellow); animation-delay: 0.2s; }
.r { background: var(--brand-red); animation-delay: 0.3s; }

@keyframes blink {
  0% { opacity: 1; }
  50% { opacity: 0.55; }
  100% { opacity: 1; }
}
@keyframes wave {
  0%,100% { transform: translateY(15px); }
  25% { transform: translateY(25px); }
  50% { transform: translateY(10px); }
  75% { transform: translateY(10px); }
}

/* ==========================================
   Login Box (Main Container)
========================================== */
.login-box {
  width: 100%;
  max-width: 420px;
  background: var(--card);
  border-radius: 20px;
  box-shadow: var(--shadow);
  border: 1px solid var(--border);
  padding: 32px 28px;
  position: relative;
  overflow: hidden;
  text-align: center;
  transition: transform 0.25s ease, box-shadow 0.25s ease;
}
.login-box:hover {
  transform: translateY(-3px);
  box-shadow: 0 10px 30px rgba(14,165,233,0.15);
}

/* ==========================================
   Welcome
========================================== */
.welcome {
  text-align: center;
  margin-bottom: 18px;
}
.welcome h1 {
  font-size: 22px;
  color: var(--brand-blue);
  font-weight: 800;
  margin: 0;
}
.welcome p {
  font-size: 14px;
  color: var(--muted);
  margin: 6px 0 0;
}

/* ==========================================
   Form Fields
========================================== */
.field {
  margin-bottom: 14px;
  text-align: right;
}
.field span {
  display: list-item;
  margin: 14px 2px 2px 10px;
  color: var(--muted);
  font-size: 14px;
  background: linear-gradient(284deg, #dadcdf, #a1bbf200);
  border-radius: 1px 7px 5px 1px;
  padding: 6px;
}
input, select, textarea {
  width: 100%;
  padding: 10px 8px;
  border: 1px solid var(--border);
  border-radius: 10px;
  background: #f8fafc;
  font-size: 14px;
  color: var(--text);
  transition: all 0.2s ease;
  box-sizing: border-box;
}
input:focus, select:focus, textarea:focus {
  border-color: var(--brand-blue);
  background: #fff;
  box-shadow: 0 0 0 4px rgba(14,165,233,.2);
  outline: none;
}

/* ==========================================
   Messages
========================================== */
.note {
  padding: 10px;
  border-radius: 10px;
  margin-bottom: 14px;
  text-align: center;
  font-size: 14px;
  font-weight: 600;
}
.note.error {
  background: #fee2e2;
  color: #b91c1c;
  border: 1px solid #fecaca;
}
.note.success {
  background: #dcfce7;
  color: #166534;
  border: 1px solid #bbf7d0;
}

/* ==========================================
   Button
========================================== */
.btn {
  width: 60%;
  border: none;
  background: linear-gradient(135deg, var(--brand-blue), #0284c7);
  color: #fff;
  border-radius: 12px;
  font-weight: 700;
  padding: 12px;
  cursor: pointer;
  font-size: 15px;
  transition: all 0.25s ease;
  box-shadow: 0 4px 12px rgba(14,165,233,.3);
  margin: 24px auto 0;
}
.btn:hover {
  transform: translateY(-2px);
}
.btn:active {
  transform: scale(0.97);
}

/* ==========================================
   Responsive
========================================== */
@media (max-width:480px){
  .login-box { padding: 24px 18px; }
  .welcome h1 { font-size: 20px; }
}
</style>
</head>

<body>

<div class="login-box">

    <div class="logo-stripes">
        <span class="b"></span>
        <span class="g"></span>
        <span class="y"></span>
        <span class="r"></span>
    </div>

    <div class="welcome">
        <h1>ثبت نام کاربر جديد</h1>
        <p>لطفا اطلاعات خود را با دقت وارد کنيد</p>
    </div>

    <?php if($msg): ?>
        <div class="note <?= $msg_type === 'success' ? 'success' : 'error' ?>"><?= $msg ?></div>
    <?php endif; ?>

    <form method="post" enctype="multipart/form-data">

        <div class="field">
            <span>نقش کاربري</span>
            <select name="role" required>
                <option value="seller"       <?= (isset($role) && $role==='seller')?'selected':'' ?>>فروشنده</option>
                <option value="buyer"        <?= (isset($role) && $role==='buyer')?'selected':'' ?>>مسئول خريد</option>
                <option value="logistic"     <?= (isset($role) && $role==='logistic')?'selected':'' ?>>لجستيک / راننده</option>
                <option value="head_carrier" <?= (isset($role) && $role==='head_carrier')?'selected':'' ?>>سرکارگر حمل</option>
                <option value="support"      <?= (isset($role) && $role==='support')?'selected':'' ?>>پشتيباني</option>
                <option value="admin"        <?= (isset($role) && $role==='admin')?'selected':'' ?>>مدير</option>
            </select>
        </div>

        <div class="field">
            <span>نام و نام خانوادگي</span>
            <input type="text" name="full_name" value="<?= htmlspecialchars($full_name ?? '') ?>" required>
        </div>

        <div class="field">
            <span>شماره موبايل (نام کاربري پيشنهادي)</span>
            <input type="text" name="mobile" value="<?= htmlspecialchars($mobile ?? '') ?>" required>
        </div>

        <div class="field">
            <span>نام کاربري پيشنهادي (در صورت تفاوت با موبايل)</span>
            <input type="text" name="username" value="<?= htmlspecialchars($username ?? '') ?>">
        </div>

        <div class="field">
            <span>کد ملي</span>
            <input type="text" name="national_id" value="<?= htmlspecialchars($national_id ?? '') ?>">
        </div>

        <div class="field">
            <span>تاريخ تولد (تقويم شمسي)</span>
            <input type="text" id="birthdate" name="birthdate" value="<?= htmlspecialchars($birth_raw ?? '') ?>">
        </div>

        <div class="field">
            <span>آدرس کامل</span>
            <textarea name="address" rows="3"><?= htmlspecialchars($address ?? '') ?></textarea>
        </div>

        <div class="field">
            <span>نوع وسيله نقليه</span>
            <select name="vehicle">
                <option value="">انتخاب کنيد</option>
                <option value="motorcycle" <?= (isset($vehicle) && $vehicle==='motorcycle')?'selected':'' ?>>موتورسيکلت</option>
                <option value="car"        <?= (isset($vehicle) && $vehicle==='car')?'selected':'' ?>>خودرو</option>
                <option value="van"        <?= (isset($vehicle) && $vehicle==='van')?'selected':'' ?>>وانت</option>
            </select>
        </div>

        <div class="field">
            <span>شماره پلاک</span>
            <input type="text" name="plate" value="<?= htmlspecialchars($plate ?? '') ?>">
        </div>

        <div class="field">
            <span>رمز عبور</span>
            <input type="password" name="password" required>
        </div>

        <div class="field">
            <span>عکس چهره راننده</span>
            <input type="file" name="photo_face" accept=".jpg,.jpeg,.png,.gif,.pdf">
        </div>

        <div class="field">
            <span>عکس کارت ملي</span>
            <input type="file" name="photo_nid" accept=".jpg,.jpeg,.png,.gif,.pdf">
        </div>

        <div class="field">
            <span>عکس کارت ماشين يا گواهينامه</span>
            <input type="file" name="photo_car" accept=".jpg,.jpeg,.png,.gif,.pdf">
        </div>

        <button class="btn">ثبت نام</button>

    </form>

</div>

<script src="https://cdn.jsdelivr.net/npm/jquery@3.7.1/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/persian-date@1.1.0/dist/persian-date.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/persian-datepicker@1.2.0/dist/js/persian-datepicker.min.js"></script>
<script>
$("#birthdate").persianDatepicker({
    format: "YYYY-MM-DD"
});
</script>

</body>
</html>
