<?php
require __DIR__.'/includes/auth.php';
require __DIR__.'/includes/config.php';
require __DIR__.'/includes/db_connect.php';
require_once __DIR__.'/includes/jdf.php';

$user     = $_SESSION['username'] ?? '';
$role     = $_SESSION['role'] ?? '';
$is_admin = ($role === 'admin');

// =========================
// Helper functions
// =========================
function nf($n){
    return number_format((int)($n ?? 0), 0, '.', ',');
}

if (!function_exists('toJalaliEn')) {
    function toJalaliEn($date){
        if(!$date || !preg_match('/^\d{4}-\d{2}-\d{2}/', $date)) return '-';
        $date = substr($date, 0, 10);
        list($gy,$gm,$gd) = explode('-', $date);
        if(!function_exists('gregorian_to_jalali')) return $date;
        return gregorian_to_jalali((int)$gy,(int)$gm,(int)$gd,'/');
    }
}

// =========================
// فیلتر ماه برای جدول فروشنده‌ها
// (برای seller استفاده می‌شود، برای admin فعلاً فقط نمودار)
// =========================
$month = $_GET['month'] ?? date('Y-m');
if (!preg_match('/^\d{4}-\d{2}$/', $month)) {
    $month = date('Y-m');
}
list($y,$m) = explode('-', $month);
$monthStart = "$y-$m-01 00:00:00";
$monthEnd   = date('Y-m-t 23:59:59', strtotime($monthStart));

// =========================
// آمار بالا + داده‌های نمودار
// =========================
$chartLabels   = [];
$chartDeposit  = [];
$chartWithdraw = [];
$chartNet      = [];

if ($is_admin) {
    // --------- آمار کلی مدیر ---------
    $totalRes = $mysqli->query("
        SELECT
          COALESCE(SUM(o.total_deposit),0) AS total_deposit,
          (SELECT COALESCE(SUM(w.amount),0) FROM withdrawals w) AS total_withdraw,
          (SELECT COALESCE(SUM(u.balance),0) FROM users u WHERE u.role='seller') AS sellers_balance,
          COUNT(DISTINCT o.id) AS orders_count,
          COUNT(DISTINCT CASE WHEN o.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN o.id END) AS orders_30d
        FROM orders o
    ");
    $totals = $totalRes ? $totalRes->fetch_assoc() : [
        'total_deposit'   => 0,
        'total_withdraw'  => 0,
        'sellers_balance' => 0,
        'orders_count'    => 0,
        'orders_30d'      => 0,
    ];

    // --------- نمودار مدیر: ۱۲ ماه اخیر ---------
    $chartQ = $mysqli->query("
        SELECT
          DATE_FORMAT(o.created_at, '%Y-%m') AS ym,
          COALESCE(SUM(o.total_deposit),0) AS total_deposit,
          COALESCE(SUM(wd.total_withdraw),0) AS total_withdraw
        FROM orders o
        LEFT JOIN (
          SELECT w.order_id, SUM(w.amount) AS total_withdraw
          FROM withdrawals w
          GROUP BY w.order_id
        ) wd ON wd.order_id = o.id
        GROUP BY DATE_FORMAT(o.created_at, '%Y-%m')
        ORDER BY ym ASC
        LIMIT 12
    ");
    if ($chartQ) {
        while($row = $chartQ->fetch_assoc()){
            $chartLabels[]   = $row['ym'];
            $chartDeposit[]  = (int)$row['total_deposit'];
            $chartWithdraw[] = (int)$row['total_withdraw'];
            $chartNet[]      = (int)$row['total_deposit'] - (int)$row['total_withdraw'];
        }
    }

    // --------- جدول فروشنده‌ها برای مدیر ---------
    $sellers = [];
    $res = $mysqli->query("
        SELECT 
          u.username,
          u.full_name,
          COALESCE(u.balance,0) AS balance,
          COALESCE(SUM(o.total_deposit),0) AS total_deposit,
          COALESCE(SUM(w.amount),0) AS total_withdraw,
          COUNT(DISTINCT o.id) AS orders_count
        FROM users u
        LEFT JOIN orders o     ON o.username = u.username
        LEFT JOIN withdrawals w ON w.order_id = o.id
        WHERE u.role = 'seller'
        GROUP BY u.username, u.full_name, u.balance
        ORDER BY u.username ASC
    ");
    if ($res) {
        while($r = $res->fetch_assoc()){
            $sellers[] = $r;
        }
    }

} else {
    // --------- آمار فروشنده ---------
    $stmt = $mysqli->prepare("
        SELECT
          COALESCE(SUM(total_deposit),0) AS total_deposit,
          (
            SELECT COALESCE(SUM(w.amount),0)
            FROM withdrawals w
            JOIN orders o2 ON o2.id=w.order_id
            WHERE o2.username=?
          ) AS total_withdraw,
          (SELECT COALESCE(balance,0) FROM users WHERE username=?) AS balance
        FROM orders o
        WHERE username=?
    ");
    $stmt->bind_param('sss', $user, $user, $user);
    $stmt->execute();
    $data = $stmt->get_result()->fetch_assoc();
    $seller_total_deposit  = (int)$data['total_deposit'];
    $seller_total_withdraw = (int)$data['total_withdraw'];
    $seller_balance        = (int)$data['balance'];

    // --------- نمودار فروشنده: ۱۲ ماه اخیر ---------
    $chartStmt = $mysqli->prepare("
        SELECT
          DATE_FORMAT(o.created_at, '%Y-%m') AS ym,
          COALESCE(SUM(o.total_deposit),0) AS total_deposit,
          COALESCE(SUM(wd.total_withdraw),0) AS total_withdraw
        FROM orders o
        LEFT JOIN (
          SELECT w.order_id, SUM(w.amount) AS total_withdraw
          FROM withdrawals w
          GROUP BY w.order_id
        ) wd ON wd.order_id = o.id
        WHERE o.username = ?
        GROUP BY DATE_FORMAT(o.created_at, '%Y-%m')
        ORDER BY ym ASC
        LIMIT 12
    ");
    $chartStmt->bind_param('s', $user);
    $chartStmt->execute();
    $chartRes = $chartStmt->get_result();
    while($row = $chartRes->fetch_assoc()){
        $chartLabels[]   = $row['ym'];
        $chartDeposit[]  = (int)$row['total_deposit'];
        $chartWithdraw[] = (int)$row['total_withdraw'];
        $chartNet[]      = (int)$row['total_deposit'] - (int)$row['total_withdraw'];
    }

    // --------- جدول تراکنش‌های ماهانه فروشنده ---------
    $txStmt = $mysqli->prepare("
      SELECT 
        o.id AS order_id,
        o.customer_name,
        o.created_at,
        o.total_deposit,
        COALESCE(SUM(w.amount),0) AS total_withdraw,
        COALESCE(o.notes,'') AS notes
      FROM orders o
      LEFT JOIN withdrawals w ON w.order_id=o.id
      WHERE o.username=? AND o.created_at BETWEEN ? AND ?
      GROUP BY o.id
      ORDER BY o.created_at DESC
    ");
    $txStmt->bind_param('sss', $user, $monthStart, $monthEnd);
    $txStmt->execute();
    $txRes = $txStmt->get_result();
    $transactions = [];
    while($r = $txRes->fetch_assoc()){
        $transactions[] = $r;
    }
}
?>
<!doctype html>
<html lang="fa" dir="rtl">
<head>
  <meta charset="utf-8">
  <title>📊 داشبورد مالی | سلام‌بابا</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link rel="stylesheet" href="assets/style.css">
  <style>
    body{
      font-family:'IRANSansWeb','IRANYekan',sans-serif;
      background:#f5f7fa;
      color:#0f172a;
      margin:0;
    }
    .container{
      max-width:1200px;
      margin:0 auto;
      padding:16px;
    }
    .page-header{
      display:flex;
      flex-wrap:wrap;
      align-items:center;
      justify-content:space-between;
      gap:10px;
      margin-bottom:16px;
    }
    .page-title{
      font-size:22px;
      font-weight:700;
      display:flex;
      align-items:center;
      gap:8px;
    }
    .page-title span.role-badge{
      font-size:11px;
      background:#e5f3ff;
      color:#0369a1;
      padding:3px 8px;
      border-radius:999px;
      border:1px solid #bae6fd;
    }
    .page-sub{
      font-size:13px;
      color:#6b7280;
    }

    .card{
      background:#fff;
      border-radius:16px;
      box-shadow:0 2px 8px rgba(15,23,42,0.06);
      padding:18px 20px;
      margin-bottom:18px;
      border:1px solid #e5e7eb;
    }
    .card-header{
      display:flex;
      justify-content:space-between;
      align-items:center;
      margin-bottom:10px;
    }
    .card-header h3{
      margin:0;
      font-size:16px;
      font-weight:600;
      display:flex;
      align-items:center;
      gap:6px;
    }
    .card-header small{
      font-size:12px;
      color:#6b7280;
    }

    .stats-grid{
      display:grid;
      grid-template-columns:repeat(4,minmax(0,1fr));
      gap:12px;
      margin-top:10px;
    }
    .stat-card{
      background:linear-gradient(135deg,#ffffff,#f9fafb);
      border-radius:14px;
      padding:14px 12px;
      box-shadow:0 1px 4px rgba(15,23,42,0.08);
      border:1px solid #e5e7eb;
      display:flex;
      flex-direction:column;
      gap:4px;
    }
    .stat-label{
      font-size:12px;
      color:#6b7280;
    }
    .stat-value{
      font-size:18px;
      font-weight:700;
      color:#0f172a;
    }
    .stat-chip{
      font-size:11px;
      align-self:flex-start;
      padding:2px 8px;
      border-radius:999px;
      border:1px solid #e5e7eb;
      color:#6b7280;
      background:#f9fafb;
    }
    .stat-value.positive{color:#16a34a;}
    .stat-value.negative{color:#dc2626;}

    .layout-two{
      display:grid;
      grid-template-columns: minmax(0,1.3fr) minmax(0,1fr);
      gap:16px;
    }

    .chart-wrapper{
      position:relative;
      min-height:260px;
    }
    #financeChart{
      width:100% !important;
      height:260px !important;
    }
    #chartEmptyMsg{
      position:absolute;
      inset:0;
      display:flex;
      align-items:center;
      justify-content:center;
      font-size:13px;
      color:#6b7280;
    }

    .table-toolbar{
      display:flex;
      flex-wrap:wrap;
      align-items:center;
      justify-content:space-between;
      gap:8px;
      margin-bottom:10px;
    }
    .toolbar-group{
      display:flex;
      flex-wrap:wrap;
      align-items:center;
      gap:6px;
    }
    .input-sm, .select-sm{
      padding:6px 8px;
      border-radius:8px;
      border:1px solid #d1d5db;
      font-size:12px;
      font-family:inherit;
      background:#fff;
      min-width:120px;
    }
    .input-sm:focus, .select-sm:focus{
      outline:none;
      border-color:#2563eb;
      box-shadow:0 0 0 2px rgba(37,99,235,0.1);
    }
    .btn-sm{
      padding:6px 10px;
      border-radius:8px;
      border:none;
      font-size:12px;
      cursor:pointer;
      font-family:inherit;
      display:inline-flex;
      align-items:center;
      gap:4px;
    }
    .btn-primary{
      background:#2563eb;
      color:#fff;
    }
    .btn-primary:hover{background:#1d4ed8;}
    .btn-ghost{
      background:#f3f4f6;
      color:#111827;
    }
    .btn-ghost:hover{background:#e5e7eb;}

    .table-wrapper{
      overflow-x:auto;
      border-radius:12px;
      border:1px solid #e5e7eb;
    }
    table{
      width:100%;
      border-collapse:collapse;
      min-width:800px;
      font-size:13px;
    }
    th,td{
      padding:8px 10px;
      text-align:center;
      border-bottom:1px solid #f1f5f9;
      white-space:nowrap;
    }
    thead th{
      background:#f8fafc;
      font-weight:600;
      color:#111827;
      cursor:pointer;
      user-select:none;
      position:relative;
    }
    tbody tr:nth-child(even){
      background:#fafafa;
    }
    tbody tr:hover{
      background:#eef2ff;
    }
    th.sort-asc::after,
    th.sort-desc::after{
      content:'';
      border:4px solid transparent;
      position:absolute;
      left:8px;
      top:50%;
      transform:translateY(-50%);
    }
    th.sort-asc::after{
      border-bottom-color:#4b5563;
    }
    th.sort-desc::after{
      border-top-color:#4b5563;
    }

    .tag-badge{
      padding:2px 6px;
      border-radius:999px;
      font-size:11px;
      border:1px solid #e5e7eb;
      background:#f9fafb;
      color:#6b7280;
    }

    .btn-table{
      font-size:12px;
      padding:3px 8px;
      border-radius:6px;
      text-decoration:none;
      display:inline-flex;
      align-items:center;
      gap:3px;
    }
    .btn-table.primary{
      background:#2563eb;
      color:#fff;
    }
    .btn-table.ghost{
      background:#f3f4f6;
      color:#111827;
    }
    .badge-balance-pos{
      color:#16a34a;
      font-weight:700;
    }
    .badge-balance-neg{
      color:#dc2626;
      font-weight:700;
    }

    .pagination{
      display:flex;
      justify-content:center;
      gap:4px;
      margin-top:8px;
      flex-wrap:wrap;
    }
    .pagination button{
      border:none;
      padding:4px 8px;
      border-radius:6px;
      background:#e5e7eb;
      color:#374151;
      font-size:12px;
      cursor:pointer;
    }
    .pagination button.active{
      background:#2563eb;
      color:#fff;
      font-weight:600;
    }
    .pagination button:hover{
      background:#dbeafe;
    }

    @media(max-width:992px){
      .layout-two{
        grid-template-columns:1fr;
      }
      .stats-grid{
        grid-template-columns:repeat(2,minmax(0,1fr));
      }
      table{font-size:12px;min-width:700px;}
    }
    @media(max-width:640px){
      .stats-grid{
        grid-template-columns:1fr 1fr;
      }
      .page-header{
        align-items:flex-start;
      }
      table{min-width:650px;}
    }
  </style>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
<div class="container">
  <?php require __DIR__.'/includes/header.php'; ?>

  <div class="page-header">
    <div>
      <div class="page-title">
        📊 داشبورد مالی
        <span class="role-badge">
          <?= $is_admin ? 'مدیر سیستم' : 'فروشنده' ?> — <?= htmlspecialchars($user) ?>
        </span>
      </div>
      <div class="page-sub">
        نمای کلی ورود و خروج پول، مانده‌ها و فاکتورهای ثبت‌شده — نسخه PRO آماده توسعه برای تحلیل سود و هزینه.
      </div>
    </div>
  </div>

  <?php if ($is_admin): ?>
    <!-- ================= آمار بالا برای مدیر ================= -->
    <section class="card">
      <div class="card-header">
        <h3>📈 آمار کلی سیستم</h3>
      </div>
      <div class="stats-grid">
        <div class="stat-card">
          <div class="stat-label">جمع کل بیعانه‌های ثبت‌شده</div>
          <div class="stat-value"><?= nf($totals['total_deposit'] ?? 0) ?> <span style="font-size:11px;">تومان</span></div>
          <div class="stat-chip">همه فروشنده‌ها</div>
        </div>
        <div class="stat-card">
          <div class="stat-label">جمع کل برداشت‌ها</div>
          <div class="stat-value"><?= nf($totals['total_withdraw'] ?? 0) ?> <span style="font-size:11px;">تومان</span></div>
          <div class="stat-chip">واریز شده به فروشنده‌ها</div>
        </div>
        <div class="stat-card">
          <?php
            $balanceSum = (int)($totals['sellers_balance'] ?? 0);
          ?>
          <div class="stat-label">مانده تجمیعی فروشنده‌ها</div>
          <div class="stat-value <?= $balanceSum >= 0 ? 'positive' : 'negative' ?>">
            <?= nf($balanceSum) ?> <span style="font-size:11px;">تومان</span>
          </div>
          <div class="stat-chip">بدهی / طلب از دید شرکت</div>
        </div>
        <div class="stat-card">
          <div class="stat-label">تعداد فاکتورهای ثبت‌شده</div>
          <div class="stat-value"><?= nf($totals['orders_count'] ?? 0) ?></div>
          <div class="stat-chip">۳۰ روز اخیر: <?= nf($totals['orders_30d'] ?? 0) ?></div>
        </div>
      </div>
    </section>

    <!-- ================= نمودار برای مدیر ================= -->
    <section class="card">
      <div class="card-header">
        <h3>📊 روند ماهانه ورود و خروج پول</h3>
        <small>نمای کلی بیعانه‌ها، برداشت‌ها و اختلاف خالص در سطح سیستم</small>
      </div>
      <div class="chart-wrapper">
        <canvas id="financeChart"></canvas>
        <div id="chartEmptyMsg" style="display:none;">داده‌ای برای نمودار یافت نشد.</div>
      </div>
    </section>

    <!-- ================= جدول فروشنده‌ها ================= -->
    <section class="card">
      <div class="card-header">
        <h3>👥 وضعیت مالی فروشنده‌ها</h3>
        <small>برای ورود به جزییات هر فروشنده روی نام کاربری کلیک کنید.</small>
      </div>

      <div class="table-toolbar">
        <div class="toolbar-group">
          <input type="text" id="sellerSearch" class="input-sm" placeholder="🔍 جستجو در نام کاربری / نام کامل">
          <select id="balanceFilter" class="select-sm">
            <option value="">همه مانده‌ها</option>
            <option value="pos">فقط مانده مثبت</option>
            <option value="neg">فقط مانده منفی</option>
            <option value="zero">فقط مانده صفر</option>
          </select>
        </div>
        <div class="toolbar-group">
          <span class="page-sub">مبنای تحلیل سود/هزینه در نسخه‌های بعدی همین جدول خواهد بود.</span>
        </div>
      </div>

      <div class="table-wrapper">
        <table id="sellerTable">
          <thead>
            <tr>
              <th data-type="text">👤 فروشنده</th>
              <th data-type="text">نام کامل</th>
              <th data-type="num">تعداد فاکتورها</th>
              <th data-type="num">جمع بیعانه‌ها</th>
              <th data-type="num">جمع برداشت‌ها</th>
              <th data-type="num">مانده حساب</th>
            </tr>
          </thead>
          <tbody>
          <?php foreach($sellers as $r): 
            $remain = (int)$r['balance'];
            $cls = $remain >= 0 ? 'badge-balance-pos' : 'badge-balance-neg';
          ?>
            <tr>
              <td>
  <a href="finance/seller.php?username=<?= urlencode($r['username']) ?>">
    <?= htmlspecialchars($r['username']) ?>
  </a>
</td>

              <td><?= htmlspecialchars($r['full_name']) ?></td>
              <td><?= nf($r['orders_count']) ?></td>
              <td><?= nf($r['total_deposit']) ?></td>
              <td><?= nf($r['total_withdraw']) ?></td>
              <td class="<?= $cls ?>"><?= nf($remain) ?></td>
            </tr>
          <?php endforeach; ?>
          <?php if (empty($sellers)): ?>
            <tr><td colspan="6">هیچ فروشنده‌ای یافت نشد.</td></tr>
          <?php endif; ?>
          </tbody>
        </table>
      </div>
      <div class="pagination" id="sellerPagination"></div>
    </section>

  <?php else: ?>
    <!-- ================= آمار بالا برای فروشنده ================= -->
    <section class="card">
      <div class="card-header">
        <h3>💼 وضعیت حساب شما</h3>
      </div>
      <div class="stats-grid">
        <div class="stat-card">
          <div class="stat-label">جمع بیعانه‌های ثبت‌شده</div>
          <div class="stat-value"><?= nf($seller_total_deposit) ?> <span style="font-size:11px;">تومان</span></div>
          <div class="stat-chip">همه فاکتورهای شما</div>
        </div>
        <div class="stat-card">
          <div class="stat-label">جمع برداشت‌های تاییدشده</div>
          <div class="stat-value"><?= nf($seller_total_withdraw) ?> <span style="font-size:11px;">تومان</span></div>
          <div class="stat-chip">تسویه‌شده با شما</div>
        </div>
        <div class="stat-card">
          <div class="stat-label">مانده حساب فعلی</div>
          <div class="stat-value <?= $seller_balance >= 0 ? 'positive' : 'negative' ?>">
            <?= nf($seller_balance) ?> <span style="font-size:11px;">تومان</span>
          </div>
          <div class="stat-chip">از دید سیستم حسابداری</div>
        </div>
        <div class="stat-card">
          <div class="stat-label">ماه انتخاب‌شده</div>
          <div class="stat-value"><?= htmlspecialchars($month) ?></div>
          <div class="stat-chip">قابل تغییر از فیلتر پایین</div>
        </div>
      </div>
    </section>

    <!-- ================= نمودار فروشنده ================= -->
    <section class="card">
      <div class="card-header">
        <h3>📊 روند ماهانه حساب شما</h3>
        <small>ورود (بیعانه)، خروج (برداشت) و خالص هر ماه</small>
      </div>
      <div class="chart-wrapper">
        <canvas id="financeChart"></canvas>
        <div id="chartEmptyMsg" style="display:none;">داده‌ای برای نمودار یافت نشد.</div>
      </div>
    </section>

    <!-- ================= جدول تراکنش‌های ماهانه فروشنده ================= -->
    <section class="card">
      <div class="card-header">
        <h3>📄 فاکتورهای این ماه</h3>
      </div>

      <form method="get" class="table-toolbar">
        <div class="toolbar-group">
          <label for="month" style="font-size:12px;color:#4b5563;">ماه:</label>
          <input type="month" id="month" name="month" value="<?= htmlspecialchars($month) ?>" class="input-sm" style="min-width:120px;">
          <button type="submit" class="btn-sm btn-primary">اعمال فیلتر</button>
        </div>
        <div class="toolbar-group">
          <input type="text" id="txSearch" class="input-sm" placeholder="🔍 جستجو در نام مشتری / توضیحات">
        </div>
      </form>

      <div class="table-wrapper">
        <table id="txTable">
          <thead>
            <tr>
              <th data-type="num">#</th>
              <th data-type="text">نام مشتری</th>
              <th data-type="text">تاریخ فاکتور</th>
              <th data-type="num">بیعانه</th>
              <th data-type="num">برداشت</th>
              <th data-type="num">مانده تجمعی</th>
              <th data-type="text">توضیحات</th>
              <th data-type="text">عملیات</th>
            </tr>
          </thead>
          <tbody>
          <?php
            $i = 1;
            $runningBalance = 0;
            foreach($transactions as $r):
              $deposit  = (int)$r['total_deposit'];
              $withdraw = (int)$r['total_withdraw'];
              $runningBalance += $deposit - $withdraw;
          ?>
            <tr>
              <td><?= $i++ ?></td>
              <td><?= htmlspecialchars($r['customer_name']) ?></td>
              <td><?= toJalaliEn(substr($r['created_at'],0,10)) ?></td>
              <td><?= nf($deposit) ?></td>
              <td><?= nf($withdraw) ?></td>
              <td class="<?= $runningBalance >= 0 ? 'badge-balance-pos' : 'badge-balance-neg' ?>">
                <?= nf($runningBalance) ?>
              </td>
              <td>
                <?php if(trim($r['notes']) !== ''): ?>
                  <span class="tag-badge" title="<?= htmlspecialchars($r['notes']) ?>">دارای توضیح</span>
                <?php else: ?>
                  <span class="tag-badge">-</span>
                <?php endif; ?>
              </td>
              <td>
                <a href="view_order.php?id=<?= (int)$r['order_id'] ?>" class="btn-table primary">👁 مشاهده</a>
                <a href="edit_order.php?id=<?= (int)$r['order_id'] ?>" class="btn-table ghost">✏️ ویرایش</a>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (empty($transactions)): ?>
            <tr><td colspan="8">📭 در این ماه هیچ تراکنشی ثبت نشده است.</td></tr>
          <?php endif; ?>
          </tbody>
        </table>
      </div>
    </section>
  <?php endif; ?>

</div>

<script>
// ====================== نمودار Chart.js ======================
(function(){
  const labels   = <?= json_encode($chartLabels, JSON_UNESCAPED_UNICODE) ?>;
  const deposits = <?= json_encode($chartDeposit, JSON_UNESCAPED_UNICODE) ?>;
  const withdraws= <?= json_encode($chartWithdraw, JSON_UNESCAPED_UNICODE) ?>;
  const netVals  = <?= json_encode($chartNet, JSON_UNESCAPED_UNICODE) ?>;

  const canvas = document.getElementById('financeChart');
  const emptyMsg = document.getElementById('chartEmptyMsg');

  if (!canvas) return;

  if (!labels.length) {
    if (emptyMsg) emptyMsg.style.display = 'flex';
    return;
  } else {
    if (emptyMsg) emptyMsg.style.display = 'none';
  }

  const ctx = canvas.getContext('2d');
  new Chart(ctx, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'بیعانه',
          data: deposits,
          borderColor: '#0ea5e9',
          backgroundColor: 'rgba(14,165,233,0.15)',
          tension: 0.3,
          borderWidth: 2,
          pointRadius: 3
        },
        {
          label: 'برداشت',
          data: withdraws,
          borderColor: '#f97316',
          backgroundColor: 'rgba(249,115,22,0.15)',
          tension: 0.3,
          borderWidth: 2,
          pointRadius: 3
        },
        {
          label: 'خالص',
          data: netVals,
          borderColor: '#22c55e',
          backgroundColor: 'rgba(34,197,94,0.15)',
          tension: 0.3,
          borderWidth: 2,
          borderDash: [4,4],
          pointRadius: 3
        }
      ]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      interaction: {
        mode: 'index',
        intersect: false
      },
      plugins: {
        legend: {
          display: true,
          labels: {
            font: { size: 11, family: 'IRANSansWeb' }
          }
        },
        tooltip: {
          callbacks: {
            label: function(ctx){
              const v = ctx.parsed.y || 0;
              return ctx.dataset.label + ': ' + v.toLocaleString('fa-IR') + ' تومان';
            }
          }
        }
      },
      scales: {
        x: {
          ticks: { font:{ size:11, family:'IRANSansWeb' } }
        },
        y: {
          ticks: {
            font:{ size:11, family:'IRANSansWeb' },
            callback: function(val){return val.toLocaleString('fa-IR');}
          },
          beginAtZero: true
        }
      }
    }
  });
})();

// ====================== جدول فروشنده‌ها (admin) ======================
(function(){
  const table = document.getElementById('sellerTable');
  if (!table) return;

  const tbody = table.querySelector('tbody');
  const rows  = Array.from(tbody.querySelectorAll('tr'));
  const searchInput  = document.getElementById('sellerSearch');
  const balanceFilter = document.getElementById('balanceFilter');
  const paginationEl = document.getElementById('sellerPagination');

  const rowsPerPage = 15;
  let currentPage = 1;
  let sortIndex = -1;
  let sortDir   = 'asc';

  function parseNum(text){
    return parseFloat(text.replace(/,/g,'').trim()) || 0;
  }

  function getFilteredSortedRows(){
    let filtered = rows.filter(r => {
      const cells = r.cells;
      if (!cells.length) return false;

      // جستجوی کلی
      const q = (searchInput.value || '').trim().toLowerCase();
      if (q){
        const hay = (cells[0].innerText + ' ' + cells[1].innerText).toLowerCase();
        if (!hay.includes(q)) return false;
      }

      // فیلتر مانده
      const bVal = balanceFilter.value;
      if (bVal){
        const balanceText = cells[5].innerText || '0';
        const balance = parseNum(balanceText);
        if (bVal === 'pos' && !(balance > 0)) return false;
        if (bVal === 'neg' && !(balance < 0)) return false;
        if (bVal === 'zero' && balance !== 0) return false;
      }

      return true;
    });

    if (sortIndex >= 0){
      const th = table.querySelectorAll('th')[sortIndex];
      const type = th.dataset.type;
      filtered.sort((a,b)=>{
        let A = a.cells[sortIndex].innerText;
        let B = b.cells[sortIndex].innerText;
        if (type === 'num'){
          A = parseNum(A); B = parseNum(B);
        } else {
          A = A.toLowerCase(); B = B.toLowerCase();
        }
        if (A === B) return 0;
        if (sortDir === 'asc') return A > B ? 1 : -1;
        return A < B ? 1 : -1;
      });
    }

    return filtered;
  }

  function render(){
    const all = getFilteredSortedRows();
    const totalPages = Math.max(1, Math.ceil(all.length / rowsPerPage));
    if (currentPage > totalPages) currentPage = totalPages;

    const start = (currentPage - 1) * rowsPerPage;
    const pageRows = all.slice(start, start + rowsPerPage);

    tbody.innerHTML = '';
    pageRows.forEach(r => tbody.appendChild(r));

    // صفحه‌بندی
    paginationEl.innerHTML = '';
    for (let i=1;i<=totalPages;i++){
      const btn = document.createElement('button');
      btn.textContent = i;
      if (i === currentPage) btn.classList.add('active');
      btn.onclick = function(){
        currentPage = i;
        render();
      };
      paginationEl.appendChild(btn);
    }
  }

  table.querySelectorAll('th').forEach((th,idx)=>{
    th.addEventListener('click', ()=>{
      if (sortIndex === idx){
        sortDir = (sortDir === 'asc') ? 'desc' : 'asc';
      } else {
        sortIndex = idx;
        sortDir = 'asc';
      }
      table.querySelectorAll('th').forEach(x=>x.classList.remove('sort-asc','sort-desc'));
      th.classList.add(sortDir === 'asc' ? 'sort-asc' : 'sort-desc');
      render();
    });
  });

  if (searchInput) searchInput.addEventListener('input', ()=>{currentPage=1;render();});
  if (balanceFilter) balanceFilter.addEventListener('change', ()=>{currentPage=1;render();});

  render();
})();

// ====================== جستجو در جدول تراکنش‌های فروشنده ======================
(function(){
  const table = document.getElementById('txTable');
  const searchInput = document.getElementById('txSearch');
  if (!table || !searchInput) return;

  const tbody = table.querySelector('tbody');
  const rows  = Array.from(tbody.querySelectorAll('tr'));

  searchInput.addEventListener('input', function(){
    const q = this.value.trim().toLowerCase();
    rows.forEach(r=>{
      const t = r.innerText.toLowerCase();
      r.style.display = (!q || t.includes(q)) ? '' : 'none';
    });
  });
})();
</script>
</body>
</html>
