<?php
/*
Plugin Name: Salam Baba Attribute Sync (Scrape → WooCommerce)
Description: Scrapes product specs from SalamBaba (PrestaShop-style pages) and syncs them into WooCommerce product attributes. Supports taxonomy mapping and scheduled updates.
Version: 1.1.0
Author: Salam Baba Ops + ChatGPT
*/

if (!defined('ABSPATH')) { exit; }

class SB_SalamBaba_Attr_Sync {

    const OPT_KEY = 'sb_sync_settings';

    public function __construct() {
        add_action('admin_menu', [$this, 'admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_post_sb_sync_now', [$this, 'handle_sync_now']);
        add_action('add_meta_boxes', [$this, 'add_meta_box']);
        add_action('save_post_product', [$this, 'save_product_meta']);
        add_action('sb_sync_cron_event', [$this, 'cron_runner']);
        add_filter('cron_schedules', [$this, 'custom_cron_schedules']);

        // Auto-sync on update (optional)
        add_action('post_updated', [$this, 'maybe_sync_on_update'], 10, 3);
    }

    /* --------------------- Settings & Admin --------------------- */

    public function admin_menu() {
        add_menu_page(
            'Salam Baba Sync',
            'Salam Baba Sync',
            'manage_woocommerce',
            'sb-salam-sync',
            [$this, 'render_settings'],
            'dashicons-updatetwo',
            57
        );
    }

    public function default_settings() {
        return [
            'table_priority'   => 'dl', // dl | table | list | auto
            'cron_enabled'     => 0,
            'cron_interval'    => 'daily',
            'overwrite'        => 1,
            'use_taxonomies'   => 1,
            'sync_price'       => 0, // risky: default off
            'sync_title'       => 0, // off
            'sync_brand_to_cat'=> 0, // also link brand as product attribute taxonomy only
            'map_json'         => json_encode([
                'مدل'                => 'model',
                'سایز (اینچ)'        => 'display_size',
                'سایز صفحه'          => 'display_size',
                'کیفیت تصویر'        => 'resolution',
                'نوع صفحه‌نمایش'     => 'panel_type',
                'نوع پنل'            => 'panel_type',
                'حداکثر رفرش‌ریت (هرتز)' => 'refresh_rate',
                'سال عرضه (میلادی)' => 'year',
                'برند'               => 'brand',
                'سیستم‌عامل'         => 'os',
                'قدرت خروجی صدا (وات)' => 'audio_power',
            ], JSON_UNESCAPED_UNICODE),
            'taxonomy_map_json'=> json_encode([
                'brand'         => 'pa_brand',
                'display_size'  => 'pa_display_size',
                'resolution'    => 'pa_resolution',
                'panel_type'    => 'pa_panel',
                'refresh_rate'  => 'pa_refresh_rate',
                'year'          => 'pa_year',
            ], JSON_UNESCAPED_UNICODE),
        ];
    }

    public function register_settings() {
        register_setting('sb_sync_group', self::OPT_KEY, function($val){
            // sanitize
            $def = $this->default_settings();
            $out = $def;
            $out['table_priority'] = in_array($val['table_priority'] ?? 'dl', ['dl','table','list','auto']) ? $val['table_priority'] : 'auto';
            $out['cron_enabled'] = !empty($val['cron_enabled']) ? 1 : 0;
            $out['cron_interval'] = in_array($val['cron_interval'] ?? 'daily', ['hourly','twicedaily','daily','every6hours']) ? $val['cron_interval'] : 'daily';
            $out['overwrite'] = !empty($val['overwrite']) ? 1 : 0;
            $out['use_taxonomies'] = !empty($val['use_taxonomies']) ? 1 : 0;
            $out['sync_price'] = !empty($val['sync_price']) ? 1 : 0;
            $out['sync_title'] = !empty($val['sync_title']) ? 1 : 0;
            $out['sync_brand_to_cat'] = !empty($val['sync_brand_to_cat']) ? 1 : 0;
            $out['map_json'] = wp_kses_post($val['map_json'] ?? $def['map_json']);
            $out['taxonomy_map_json'] = wp_kses_post($val['taxonomy_map_json'] ?? $def['taxonomy_map_json']);
            return $out;
        });

        $opts = $this->get_settings();
        $this->maybe_register_or_clear_cron($opts);
    }

    public function custom_cron_schedules($schedules) {
        if (!isset($schedules['every6hours'])) {
            $schedules['every6hours'] = [
                'interval' => 6 * HOUR_IN_SECONDS,
                'display'  => __('Every 6 Hours', 'sb-sync')
            ];
        }
        return $schedules;
    }

    public function get_settings() {
        $saved = get_option(self::OPT_KEY, []);
        return wp_parse_args($saved, $this->default_settings());
    }

    public function render_settings() {
        if (!current_user_can('manage_woocommerce')) return;
        $o = $this->get_settings();
        ?>
        <div class="wrap">
            <h1>Salam Baba Attribute Sync</h1>
            <p>Scrape ساختار PrestaShop (سلام‌بابا) از <code>dl.data-sheet</code> و تبدیل به ویژگی‌های ووکامرس. امکان مپینگ فارسی→کلید و کلید→taxonomy.</p>

            <form method="post" action="options.php">
                <?php settings_fields('sb_sync_group'); ?>
                <table class="form-table">
                    <tr>
                        <th>Priority</th>
                        <td>
                            <select name="<?php echo self::OPT_KEY; ?>[table_priority]">
                                <?php foreach(['dl'=>'Definition List (dl/dt/dd)','table'=>'Table (tr/th/td)','list'=>'Lists (li)','auto'=>'Auto'] as $k=>$lbl): ?>
                                    <option value="<?php echo esc_attr($k); ?>" <?php selected($o['table_priority'],$k); ?>><?php echo esc_html($lbl); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th>Overwrite attributes?</th>
                        <td><label><input type="checkbox" name="<?php echo self::OPT_KEY; ?>[overwrite]" value="1" <?php checked($o['overwrite'],1); ?> /> Yes</label></td>
                    </tr>
                    <tr>
                        <th>Use Taxonomies</th>
                        <td><label><input type="checkbox" name="<?php echo self::OPT_KEY; ?>[use_taxonomies]" value="1" <?php checked($o['use_taxonomies'],1); ?> /> Convert mapped keys to product attribute taxonomies</label></td>
                    </tr>
                    <tr>
                        <th>Cron</th>
                        <td>
                            <label><input type="checkbox" name="<?php echo self::OPT_KEY; ?>[cron_enabled]" value="1" <?php checked($o['cron_enabled'],1); ?> /> Enabled</label>
                            <select name="<?php echo self::OPT_KEY; ?>[cron_interval]">
                                <?php foreach(['hourly'=>'Hourly','twicedaily'=>'Twice Daily','every6hours'=>'Every 6 hours','daily'=>'Daily'] as $k=>$lbl): ?>
                                    <option value="<?php echo esc_attr($k); ?>" <?php selected($o['cron_interval'],$k); ?>><?php echo esc_html($lbl); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th>Also sync (risky)</th>
                        <td>
                            <label><input type="checkbox" name="<?php echo self::OPT_KEY; ?>[sync_price]" value="1" <?php checked($o['sync_price'],1); ?> /> Price</label>
                            &nbsp;&nbsp;
                            <label><input type="checkbox" name="<?php echo self::OPT_KEY; ?>[sync_title]" value="1" <?php checked($o['sync_title'],1); ?> /> Title</label>
                        </td>
                    </tr>
                    <tr>
                        <th>Label → Key (JSON)</th>
                        <td>
                            <textarea name="<?php echo self::OPT_KEY; ?>[map_json]" rows="8" style="width:100%; direction:ltr;"><?php echo esc_textarea($o['map_json']); ?></textarea>
                            <p class="description">کلیدهای فارسی را به کلیدهای لاتین تبدیل می‌کند.</p>
                        </td>
                    </tr>
                    <tr>
                        <th>Key → Taxonomy (JSON)</th>
                        <td>
                            <textarea name="<?php echo self::OPT_KEY; ?>[taxonomy_map_json]" rows="5" style="width:100%; direction:ltr;"><?php echo esc_textarea($o['taxonomy_map_json']); ?></textarea>
                            <p class="description">برای هر کلید، نام taxonomy مربوطه مثل <code>pa_brand</code> را بنویس.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Save Settings'); ?>
            </form>

            <hr/>
            <h2>Manual Sync</h2>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <?php wp_nonce_field('sb_sync_now'); ?>
                <input type="hidden" name="action" value="sb_sync_now" />
                <table class="form-table">
                    <tr>
                        <th>Product ID</th>
                        <td><input type="number" name="product_id" required class="regular-text"/></td>
                    </tr>
                    <tr>
                        <th>Source URL</th>
                        <td><input type="url" name="source_url" required class="regular-text" placeholder="https://saalambabaa.com/SBP-7853/sony"/></td>
                    </tr>
                </table>
                <?php submit_button('Scrape & Sync'); ?>
            </form>
        </div>
        <?php
    }

    /* --------------------- Meta Box --------------------- */

    public function add_meta_box() {
        add_meta_box('sb_salam_sync', 'SB SalamBaba Sync', [$this,'render_meta_box'], 'product', 'side', 'high');
    }

    public function render_meta_box($post) {
        $url = get_post_meta($post->ID, '_sb_source_url', true);
        $sync_on_update = get_post_meta($post->ID, '_sb_sync_on_update', true);
        wp_nonce_field('sb_meta_save','sb_meta_nonce');
        ?>
        <p><label>Source URL</label>
        <input type="url" name="sb_source_url" value="<?php echo esc_attr($url); ?>" style="width:100%" placeholder="https://saalambabaa.com/..."/></p>
        <p><label><input type="checkbox" name="sb_sync_on_update" value="1" <?php checked($sync_on_update, '1'); ?>/> Sync on update</label></p>
        <p class="description">اگر فعال باشد، با هر بار ذخیره محصول، مشخصات از منبع بازخوانی می‌شود.</p>
        <?php
    }

    public function save_product_meta($post_id) {
        if (!isset($_POST['sb_meta_nonce']) || !wp_verify_nonce($_POST['sb_meta_nonce'],'sb_meta_save')) return;
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        if (!current_user_can('edit_product', $post_id)) return;

        $url = isset($_POST['sb_source_url']) ? esc_url_raw($_POST['sb_source_url']) : '';
        $sync_on_update = !empty($_POST['sb_sync_on_update']) ? '1' : '0';

        if ($url) update_post_meta($post_id, '_sb_source_url', $url);
        else delete_post_meta($post_id, '_sb_source_url');

        update_post_meta($post_id, '_sb_sync_on_update', $sync_on_update);
    }

    /* --------------------- Cron --------------------- */

    private function maybe_register_or_clear_cron($opts) {
        $hook = 'sb_sync_cron_event';
        $ts = wp_next_scheduled($hook);
        if ($opts['cron_enabled']) {
            if (!$ts) wp_schedule_event(time() + 120, $opts['cron_interval'], $hook);
        } else {
            if ($ts) wp_clear_scheduled_hook($hook);
        }
    }

    public function cron_runner() {
        $q = new WP_Query([
            'post_type'=>'product',
            'posts_per_page'=>-1,
            'fields'=>'ids',
            'meta_query'=>[['key'=>'_sb_source_url','compare'=>'EXISTS']]
        ]);
        if ($q->have_posts()) {
            foreach ($q->posts as $pid) {
                $url = get_post_meta($pid, '_sb_source_url', true);
                if ($url) { $this->sync_product($pid, $url); }
            }
        }
    }

    public function maybe_sync_on_update($post_ID, $post_after, $post_before) {
        if ($post_after->post_type !== 'product') return;
        $flag = get_post_meta($post_ID, '_sb_sync_on_update', true);
        $url  = get_post_meta($post_ID, '_sb_source_url', true);
        if ($flag === '1' && $url) { $this->sync_product($post_ID, $url); }
    }

    /* --------------------- Sync Flow --------------------- */

    public function handle_sync_now() {
        if (!current_user_can('manage_woocommerce')) wp_die('No permission');
        check_admin_referer('sb_sync_now');
        $pid = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
        $url = isset($_POST['source_url']) ? esc_url_raw($_POST['source_url']) : '';
        $res = $this->sync_product($pid, $url);
        $msg = $res['ok'] ? 'ok' : 'fail';
        wp_redirect(add_query_arg(['page'=>'sb-salam-sync','sb_msg'=>$msg,'sb_detail'=>rawurlencode($res['msg'])], admin_url('admin.php')));
        exit;
    }

    private function http_get($url) {
        $resp = wp_remote_get($url, [
            'timeout' => 25,
            'headers' => ['User-Agent'=>'SB-SalamBaba-Sync/1.1 (+https://slmb.ir)']
        ]);
        if (is_wp_error($resp)) return ['ok'=>false,'code'=>0,'body'=>'','msg'=>$resp->get_error_message()];
        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);
        if ($code !== 200 || empty($body)) return ['ok'=>false,'code'=>$code,'body'=>'','msg'=>"HTTP $code"];
        return ['ok'=>true,'code'=>$code,'body'=>$body,'msg'=>'ok'];
    }

    private function normalize($str) {
        $str = wp_strip_all_tags($str);
        $str = html_entity_decode($str, ENT_QUOTES, 'UTF-8');
        $str = preg_replace('/\s+/u',' ', $str);
        $str = trim($str);
        return $str;
    }

    private function fa_to_en_digits($s) {
        $fa = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹','٫','٬'];
        $en = ['0','1','2','3','4','5','6','7','8','9','.',''];
        return str_replace($fa,$en,$s);
    }

    private function slugify($txt) {
        $map = [
            ' '=>'-','ـ'=>'-','_'=>'-','/'=>'-','\\'=>'-','|'=>'-','.'=>'-','،'=>'-','؛'=>'-','«'=>'','»'=>'','('=>'',')'=>'','['=>'',']'=>'','{' => '', '}' => '',
            'آ'=>'a','ا'=>'a','ب'=>'b','پ'=>'p','ت'=>'t','ث'=>'s','ج'=>'j','چ'=>'ch','ح'=>'h','خ'=>'kh','د'=>'d','ذ'=>'z','ر'=>'r','ز'=>'z','ژ'=>'zh','س'=>'s','ش'=>'sh','ص'=>'s','ض'=>'z','ط'=>'t','ظ'=>'z','ع'=>'a','غ'=>'gh','ف'=>'f','ق'=>'gh','ک'=>'k','گ'=>'g','ل'=>'l','م'=>'m','ن'=>'n','و'=>'v','ه'=>'h','ی'=>'y',
            'ء'=>'','‍'=>''
        ];
        $txt = strtr($txt, $map);
        $txt = strtolower($txt);
        $txt = preg_replace('/[^a-z0-9\-]+/','', $txt);
        $txt = preg_replace('/-+/','-', $txt);
        return trim($txt,'-');
    }

    private function parse_spec_pairs($html, $priority='dl') {
        $pairs = [];

        if (mb_detect_encoding($html, 'UTF-8', true) === false) {
            $html = mb_convert_encoding($html, 'HTML-ENTITIES', 'UTF-8');
        }

        libxml_use_internal_errors(true);
        $dom = new DOMDocument();
        @$dom->loadHTML($html);
        $xpath = new DOMXPath($dom);

        $order = ($priority==='auto') ? ['dl','table','list'] : [$priority,'table','list'];
        $order = array_unique($order);

        foreach ($order as $mode) {
            if ($mode==='dl') {
                // PrestaShop style: dl.data-sheet > dt.name / dd.value
                $dts = $xpath->query('//dl[contains(@class,"data-sheet")]//dt');
                foreach ($dts as $dt) {
                    $dd = $dt->nextSibling;
                    while ($dd && ($dd->nodeType !== XML_ELEMENT_NODE || strtolower($dd->nodeName)!=='dd')) {
                        $dd = $dd->nextSibling;
                    }
                    if ($dd) {
                        $k = $this->normalize($dt->textContent);
                        $v = $this->normalize($dd->textContent);
                        if ($k && $v) $pairs[$k] = $v;
                    }
                }
            } elseif ($mode==='table') {
                $trs = $xpath->query('//table//tr');
                foreach ($trs as $tr) {
                    $th = null; $td = null;
                    foreach ($tr->childNodes as $c) {
                        if ($c->nodeType === XML_ELEMENT_NODE) {
                            if (!$th && strtolower($c->nodeName)==='th') $th = $c;
                            if (!$td && strtolower($c->nodeName)==='td') $td = $c;
                        }
                    }
                    if ($th && $td) {
                        $k = $this->normalize($th->textContent);
                        $v = $this->normalize($td->textContent);
                        if ($k && $v) $pairs[$k] = $v;
                    }
                }
            } else { // list
                $lis = $xpath->query('//li');
                foreach ($lis as $li) {
                    $txt = $this->normalize($li->textContent);
                    if (strpos($txt, ':') !== false) {
                        $tmp = explode(':',$txt,2);
                        $k = trim($tmp[0]); $v = trim($tmp[1]);
                        if ($k && $v) $pairs[$k] = $v;
                    }
                }
            }
            if (!empty($pairs)) break;
        }

        // basic cleanup (length limits)
        $clean = [];
        foreach ($pairs as $k=>$v) {
            $k = preg_replace('/[:：]+$/u','',$k);
            $k = trim($k);
            $v = trim($v);
            if (mb_strlen($k)<=120 && mb_strlen($v)<=1000) $clean[$k]=$v;
        }
        return $clean;
    }

    private function map_labels_to_keys($pairs, $labelMapJson) {
        $out = [];
        $labelMap = json_decode($labelMapJson, true) ?: [];
        foreach ($pairs as $label=>$val) {
            $label_norm = $this->normalize($label);
            $key = $labelMap[$label_norm] ?? $labelMap[$label] ?? $this->slugify($label_norm);
            $out[$key] = $this->normalize($this->fa_to_en_digits($val));
        }
        return $out;
    }

    private function ensure_attribute_taxonomy($taxonomy, $label) {
        if (taxonomy_exists($taxonomy)) return true;
        if (!function_exists('wc_create_attribute')) return false;
        $args = [
            'slug'         => $taxonomy,
            'name'         => $label,
            'type'         => 'select',
            'order_by'     => 'menu_order',
            'has_archives' => false,
        ];
        $id = wc_create_attribute($args);
        if (is_wp_error($id) || !$id) return false;
        // Flush rules so that the new taxonomy is registered
        delete_transient('wc_attribute_taxonomies');
        wc_get_attribute_taxonomies();
        return true;
    }

    private function attach_tax_terms($product, $taxonomy, $value) {
        $value = trim($value);
        if (!$value) return;
        if (!taxonomy_exists($taxonomy)) return;

        // Find or create term
        $term = term_exists($value, $taxonomy);
        if (!$term) {
            $term = wp_insert_term($value, $taxonomy, ['slug'=>$this->slugify($value)]);
            if (is_wp_error($term)) return;
        }
        $term_id = is_array($term) ? $term['term_id'] : $term;

        // Set terms
        wp_set_object_terms($product->get_id(), intval($term_id), $taxonomy, false);

        // Link attribute to product visible attributes
        $attrs = $product->get_attributes();
        if (!is_array($attrs)) $attrs = [];
        $wp_attr = new WC_Product_Attribute();
        $wp_attr->set_id( wc_attribute_taxonomy_id_by_name($taxonomy) );
        $wp_attr->set_name( $taxonomy );
        $wp_attr->set_options( [ $term_id ] );
        $wp_attr->set_visible(true);
        $wp_attr->set_variation(false);
        $attrs[] = $wp_attr;
        $product->set_attributes($attrs);
    }

    private function set_custom_attributes($product, $assoc, $overwrite = true) {
        $attrs = $product->get_attributes();
        if (!is_array($attrs)) $attrs = [];
        if ($overwrite) $attrs = [];
        $pos = 0;
        foreach ($assoc as $k=>$v) {
            $pa = new WC_Product_Attribute();
            $pa->set_name($k);
            $pa->set_options([ $v ]);
            $pa->set_visible(true);
            $pa->set_variation(false);
            $pa->set_position($pos++);
            $attrs[] = $pa;
        }
        $product->set_attributes($attrs);
    }

    private function maybe_sync_price($product, $html) {
        // naive: try to find price-like patterns e.g., data-price, itemprop="price", or numbers with currency
        // This is intentionally conservative.
        if (!function_exists('wc_get_price_decimals')) return;
        $m = [];
        if (preg_match('/itemprop=["\']price["\'][^>]*content=["\']([\d\.]+)/u', $html, $m)) {
            $price = floatval($m[1]);
        } elseif (preg_match('/data-price=["\']([\d\.]+)/u', $html, $m)) {
            $price = floatval($m[1]);
        } else {
            return; // not found
        }
        if ($price > 0) {
            $product->set_regular_price($price);
            if (!$product->get_price()) $product->set_price($price);
        }
    }

    public function sync_product($product_id, $source_url) {
        if (!function_exists('wc_get_product')) return ['ok'=>false,'msg'=>'WooCommerce not active'];
        $product = wc_get_product($product_id);
        if (!$product) return ['ok'=>false,'msg'=>'Invalid product ID'];

        $opts = $this->get_settings();
        if (!$source_url) $source_url = get_post_meta($product_id, '_sb_source_url', true);
        if (!$source_url) return ['ok'=>false,'msg'=>'Missing source URL'];

        $http = $this->http_get($source_url);
        if (!$http['ok']) return ['ok'=>false,'msg'=>'Fetch failed: '.$http['msg']];

        $pairs = $this->parse_spec_pairs($http['body'], $opts['table_priority']);
        if (empty($pairs)) return ['ok'=>false,'msg'=>'No specs found in HTML'];

        $assoc = $this->map_labels_to_keys($pairs, $opts['map_json']);

        // Taxonomies handling
        $taxMap = json_decode($opts['taxonomy_map_json'], true) ?: [];
        $usedTax = 0;
        if (!empty($opts['use_taxonomies'])) {
            foreach ($taxMap as $key=>$tax) {
                if (!empty($assoc[$key])) {
                    // ensure taxonomy exists
                    $this->ensure_attribute_taxonomy($tax, strtoupper(str_replace('pa_','',$tax)));
                    $this->attach_tax_terms($product, $tax, $assoc[$key]);
                    unset($assoc[$key]);
                    $usedTax++;
                }
            }
        }

        // Custom attributes for leftovers
        if (!empty($assoc)) {
            $this->set_custom_attributes($product, $assoc, !empty($opts['overwrite']));
        }

        // Optional risky syncing
        if (!empty($opts['sync_price'])) $this->maybe_sync_price($product, $http['body']);
        if (!empty($opts['sync_title'])) {
            // VERY conservative: set the title from "مدل" if exists
            if (!empty($pairs['مدل'])) {
                $post_arr = ['ID'=>$product_id, 'post_title'=> sanitize_text_field($pairs['مدل'])];
                wp_update_post($post_arr);
            }
        }

        $product->save();
        update_post_meta($product_id, '_sb_source_url', esc_url_raw($source_url));

        $msg = sprintf('Synced %d specs (%d via taxonomy, %d as custom).',
            count($pairs), $usedTax, count($pairs) - $usedTax);
        return ['ok'=>true,'msg'=>$msg];
    }
}

new SB_SalamBaba_Attr_Sync();
